class UsersManager {
    constructor() {
        this.users = [];
        this.filteredUsers = [];
        this.currentPage = 1;
        this.usersPerPage = 10;
        this.currentSort = 'date';
        this.sortDirection = 'desc';
        
        this.initializeEventListeners();
        this.loadUsers();
        this.startRealTimeUpdates();
    }

    initializeEventListeners() {
        // Search
        document.getElementById('searchInput').addEventListener('input', (e) => {
            this.filterUsers(e.target.value);
        });

        // Sorting
        document.getElementById('sortSelect').addEventListener('change', (e) => {
            this.currentSort = e.target.value;
            this.sortUsers();
        });

        // Refresh data
        document.getElementById('refreshBtn').addEventListener('click', () => {
            this.loadUsers();
        });

        // Sorting when clicking table headers
        document.querySelectorAll('th[data-sort]').forEach(th => {
            th.addEventListener('click', () => {
                const sortField = th.getAttribute('data-sort');
                this.sortUsers(sortField);
            });
        });

        // Pagination
        document.getElementById('prevBtn').addEventListener('click', () => {
            this.previousPage();
        });

        document.getElementById('nextBtn').addEventListener('click', () => {
            this.nextPage();
        });
    }

    async loadUsers() {
        try {
            console.log("🔗 Connecting to Firestore...");
            this.showLoading(true);
            
            // Get all users
            const querySnapshot = await db.collection("users").get();
            console.log("📊 Total users in database:", querySnapshot.size);
            
            this.users = [];
            
            querySnapshot.forEach((doc) => {
                const userData = doc.data();
                console.log("👤 User:", doc.id, "=>", userData);
                
                // Calculate activity status based on login status
                const isLoggedIn = userData.isLoggedIn === true;
                const lastLogin = userData.lastLogin;
                const lastLogout = userData.lastLogout;
                
                this.users.push({
                    id: doc.id,
                    name: userData.name || userData.displayName || userData.username || userData.fullName || userData.firstName || userData.userName || userData.Name || 'Not specified',
                    email: userData.email || userData.Email || userData.userEmail || 'Not specified',
                    phone: userData.phone || userData.phoneNumber || userData.mobile || userData.Phone || userData.telephone || 'Not specified',
                    country: userData.country || userData.Country || userData.userCountry || userData.nationality || userData.location || 'Not specified',
                    createdAt: userData.createdAt || userData.registrationDate || userData.signupDate || userData.timestamp || userData.createdDate,
                    isActive: isLoggedIn,
                    lastLogin: lastLogin,
                    lastLogout: lastLogout,
                    isLoggedIn: isLoggedIn,
                    rawData: userData
                });
            });

            console.log("📋 Processed users:", this.users);
            this.filteredUsers = [...this.users];
            this.sortUsers();
            this.updateStats();
            this.showLoading(false);
            
        } catch (error) {
            console.error("❌ Error loading data:", error);
            this.showError("Error loading data: " + error.message);
            this.showLoading(false);
        }
    }

    startRealTimeUpdates() {
        // Listen for real-time changes in Firestore
        db.collection("users").onSnapshot((snapshot) => {
            console.log("🔄 Real-time update received! Reloading data...");
            this.loadUsers();
        }, (error) => {
            console.error("❌ Real-time updates error:", error);
        });
    }

    filterUsers(searchTerm) {
        if (!searchTerm) {
            this.filteredUsers = [...this.users];
        } else {
            const term = searchTerm.toLowerCase();
            this.filteredUsers = this.users.filter(user => 
                (user.name && user.name.toLowerCase().includes(term)) ||
                (user.email && user.email.toLowerCase().includes(term)) ||
                (user.phone && user.phone.toLowerCase().includes(term)) ||
                (user.country && user.country.toLowerCase().includes(term)) ||
                (user.id && user.id.toLowerCase().includes(term))
            );
        }
        
        this.currentPage = 1;
        this.renderUsers();
    }

    sortUsers(field = this.currentSort) {
        if (field === this.currentSort) {
            this.sortDirection = this.sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            this.currentSort = field;
            this.sortDirection = 'asc';
        }

        this.filteredUsers.sort((a, b) => {
            let aValue = a[field];
            let bValue = b[field];

            // Handle empty values
            if (!aValue && !bValue) return 0;
            if (!aValue) return this.sortDirection === 'asc' ? -1 : 1;
            if (!bValue) return this.sortDirection === 'asc' ? 1 : -1;

            // Handle dates
            if (field === 'date' || field === 'createdAt' || field === 'lastLogin' || field === 'lastLogout') {
                if (aValue && aValue.seconds) aValue = aValue.seconds * 1000;
                else if (aValue && typeof aValue === 'string') aValue = new Date(aValue).getTime();
                else aValue = 0;
                
                if (bValue && bValue.seconds) bValue = bValue.seconds * 1000;
                else if (bValue && typeof bValue === 'string') bValue = new Date(bValue).getTime();
                else bValue = 0;
            }

            // Handle boolean values (login status)
            if (field === 'isLoggedIn' || field === 'isActive') {
                aValue = aValue ? 1 : 0;
                bValue = bValue ? 1 : 0;
            }

            // Handle text
            if (typeof aValue === 'string') aValue = aValue.toLowerCase();
            if (typeof bValue === 'string') bValue = bValue.toLowerCase();

            if (aValue < bValue) return this.sortDirection === 'asc' ? -1 : 1;
            if (aValue > bValue) return this.sortDirection === 'asc' ? 1 : -1;
            return 0;
        });

        this.renderUsers();
        this.updateSortIndicators();
    }

    updateSortIndicators() {
        document.querySelectorAll('th[data-sort]').forEach(th => {
            const field = th.getAttribute('data-sort');
            let text = th.textContent.replace(' ⬍', '').replace(' ⬌', '');
            th.textContent = text;
            
            if (field === this.currentSort) {
                th.textContent = text + (this.sortDirection === 'asc' ? ' ⬍' : ' ⬌');
            }
        });
    }

    renderUsers() {
        const tbody = document.getElementById('usersTableBody');
        tbody.innerHTML = '';

        const startIndex = (this.currentPage - 1) * this.usersPerPage;
        const endIndex = startIndex + this.usersPerPage;
        const usersToShow = this.filteredUsers.slice(startIndex, endIndex);

        if (usersToShow.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="7" style="text-align: center; padding: 40px; color: var(--gray-light);">
                        ${this.users.length === 0 ? 'No user data available' : 'No matching results found'}
                    </td>
                </tr>
            `;
            return;
        }

        usersToShow.forEach(user => {
            const row = tbody.insertRow();
            row.className = 'fade-in-up';
            
            // Format date
            let dateString = 'Not specified';
            if (user.createdAt) {
                if (user.createdAt.seconds) {
                    dateString = new Date(user.createdAt.seconds * 1000).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'short',
                        day: 'numeric'
                    });
                } else {
                    dateString = new Date(user.createdAt).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'short',
                        day: 'numeric'
                    });
                }
            }

            row.innerHTML = `
                <td>${this.escapeHtml(user.name)}</td>
                <td>${this.escapeHtml(user.email)}</td>
                <td>${this.escapeHtml(user.phone)}</td>
                <td>${this.escapeHtml(user.country)}</td>
                <td>${dateString}</td>
                <td>
                    <span class="${user.isLoggedIn ? 'status-active' : 'status-inactive'}">
                        ${user.isLoggedIn ? 'Online' : 'Offline'}
                    </span>
                </td>
                <td>
                    <button class="btn-view" onclick="usersManager.viewUserDetails('${user.id}')">
                        View Details
                    </button>
                </td>
            `;
        });

        this.updatePagination();
    }

    updatePagination() {
        const totalPages = Math.ceil(this.filteredUsers.length / this.usersPerPage);
        const pageInfo = document.getElementById('pageInfo');
        const prevBtn = document.getElementById('prevBtn');
        const nextBtn = document.getElementById('nextBtn');

        pageInfo.textContent = `Page ${this.currentPage} of ${totalPages}`;
        prevBtn.disabled = this.currentPage === 1;
        nextBtn.disabled = this.currentPage === totalPages || totalPages === 0;
    }

    previousPage() {
        if (this.currentPage > 1) {
            this.currentPage--;
            this.renderUsers();
        }
    }

    nextPage() {
        const totalPages = Math.ceil(this.filteredUsers.length / this.usersPerPage);
        if (this.currentPage < totalPages) {
            this.currentPage++;
            this.renderUsers();
        }
    }

    updateStats() {
        const totalUsers = this.users.length;
        const onlineUsers = this.users.filter(user => user.isLoggedIn).length;
        const offlineUsers = totalUsers - onlineUsers;
        
        document.getElementById('totalUsers').textContent = totalUsers;
        document.getElementById('activeUsers').textContent = onlineUsers;
        document.getElementById('offlineUsers').textContent = offlineUsers;

        console.log("📈 Statistics:", { 
            totalUsers, 
            onlineUsers, 
            offlineUsers 
        });
    }

    viewUserDetails(userId) {
        const user = this.users.find(u => u.id === userId);
        if (user) {
            let details = `User Details:\n\n`;
            details += `🆔 ID: ${user.id}\n`;
            details += `👤 Name: ${user.name}\n`;
            details += `📧 Email: ${user.email}\n`;
            details += `📞 Phone: ${user.phone}\n`;
            details += `🌍 Country: ${user.country}\n`;
            
            if (user.createdAt) {
                let dateString = 'Not specified';
                if (user.createdAt.seconds) {
                    dateString = new Date(user.createdAt.seconds * 1000).toLocaleString('en-US');
                } else {
                    dateString = new Date(user.createdAt).toLocaleString('en-US');
                }
                details += `📅 Registration Date: ${dateString}\n`;
            }
            
            details += `🔐 Login Status: ${user.isLoggedIn ? '✅ Logged In' : '❌ Logged Out'}\n`;
            
            // Login time information
            if (user.lastLogin && user.isLoggedIn) {
                let lastLoginString = 'Not specified';
                if (user.lastLogin.seconds) {
                    lastLoginString = new Date(user.lastLogin.seconds * 1000).toLocaleString('en-US');
                } else {
                    lastLoginString = new Date(user.lastLogin).toLocaleString('en-US');
                }
                details += `🕒 Last Login: ${lastLoginString}\n`;
            }
            
            if (user.lastLogout && !user.isLoggedIn) {
                let lastLogoutString = 'Not specified';
                if (user.lastLogout.seconds) {
                    lastLogoutString = new Date(user.lastLogout.seconds * 1000).toLocaleString('en-US');
                } else {
                    lastLogoutString = new Date(user.lastLogout).toLocaleString('en-US');
                }
                details += `🚪 Last Logout: ${lastLogoutString}\n`;
            }

            alert(details);
        }
    }

    showLoading(show) {
        const loading = document.getElementById('loading');
        const tableContainer = document.querySelector('.table-container');
        
        if (show) {
            loading.style.display = 'block';
            loading.classList.add('active');
            if (tableContainer) tableContainer.style.display = 'none';
        } else {
            loading.style.display = 'none';
            loading.classList.remove('active');
            if (tableContainer) tableContainer.style.display = 'block';
        }
    }

    showError(message) {
        const existingError = document.querySelector('.error-message');
        if (existingError) {
            existingError.remove();
        }

        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.textContent = message;

        document.body.appendChild(errorDiv);

        setTimeout(() => {
            if (errorDiv.parentNode) {
                errorDiv.parentNode.removeChild(errorDiv);
            }
        }, 5000);
    }

    escapeHtml(unsafe) {
        if (unsafe === null || unsafe === undefined) return 'Not specified';
        if (typeof unsafe !== 'string') return unsafe.toString();
        
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
}

// Initialize the users manager when the page loads
let usersManager;

document.addEventListener('DOMContentLoaded', () => {
    // Check Firebase initialization first
    if (typeof firebase !== 'undefined' && firebase.apps.length > 0) {
        usersManager = new UsersManager();
    } else {
        console.error("❌ Firebase not initialized!");
        document.body.innerHTML = `
            <div style="text-align: center; padding: 50px; color: var(--danger);">
                <h2>❌ Initialization Error</h2>
                <p>Firebase is not properly initialized</p>
                <p>Check firebase-config.js file</p>
            </div>
        `;
    }
});

// Make manager globally accessible
window.usersManager = usersManager;