const API_BASE = '../api';

class AdminApp {
    constructor() {
        this.categories = [];
        this.books = [];
        this.profileCategories = [];
        
        // ?? NEW: Pagination properties
        this.currentBookPage = 1;
        this.booksPerPage = 50;
        this.totalBookPages = 1;
        
        this.currentCategoryPage = 1;
        this.categoriesPerPage = 20;
        this.totalCategoryPages = 1;
        
        this.currentProfileCategoryPage = 1;
        this.profileCategoriesPerPage = 20;
        this.totalProfileCategoryPages = 1;
        
        this.init();
    }
    
    logout() {
    if (confirm('Are you sure you want logout?')) {
        // ??? ??????? ??????
        localStorage.clear();
        sessionStorage.clear();
        
        // ??????? ????? ????? ??????
        window.location.href = 'logout.php';
    }
}

// Updated function using the new endpoint
async updateGlobalAuthSettings() {
    try {
        console.log('?? Updating global download authentication settings...');
        
        const authOption = document.querySelector('input[name="global_auth_option"]:checked').value;
        const isConfirmed = document.getElementById('global-auth-confirm').checked;
        
        if (!isConfirmed) {
            alert('Please confirm that you understand this will modify ALL books');
            return;
        }
        
        const enableAuth = authOption === 'enable';
        
        if (!confirm(`Are you sure you want to ${enableAuth ? 'ENABLE' : 'DISABLE'} download authentication for ALL ${this.books.length} books? This action cannot be undone.`)) {
            return;
        }
        
        console.log(`?? Setting download authentication to: ${enableAuth ? 'ENABLED' : 'DISABLED'} for all books`);
        
        const response = await fetch(`${API_BASE}/global_operations.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'update_global_auth',
                require_login_for_download: enableAuth
            })
        });

        const result = await response.json();

        if (!response.ok) {
            throw new Error(result.error || 'Server error');
        }

        alert(`Successfully ${enableAuth ? 'enabled' : 'disabled'} download authentication for ${result.updated_books} books`);
        
        // Reload books to reflect changes
        await this.loadBooks();
        this.updateGlobalAuthStats();
        
    } catch (error) {
        console.error('Error updating global auth settings:', error);
        alert('Error while updating: ' + error.message);
    }
}

// ?? NEW: Update global auth statistics
updateGlobalAuthStats() {
    const totalBooks = this.books.length;
    const authEnabledCount = this.books.filter(book => 
        book.require_login_for_download === true || book.require_login_for_download === 1 || book.require_login_for_download === '1'
    ).length;
    const authDisabledCount = totalBooks - authEnabledCount;

    // Update statistics
    const totalElement = document.getElementById('total-books-auth');
    const enabledElement = document.getElementById('auth-enabled-count');
    const disabledElement = document.getElementById('auth-disabled-count');
    
    if (totalElement) totalElement.textContent = totalBooks;
    if (enabledElement) enabledElement.textContent = authEnabledCount;
    if (disabledElement) disabledElement.textContent = authDisabledCount;

    console.log(`?? Global Auth Stats - Total: ${totalBooks}, Enabled: ${authEnabledCount}, Disabled: ${authDisabledCount}`);
}

// ?? NEW: Load global auth section
loadGlobalAuthSection() {
    this.updateGlobalAuthStats();
}

    init() {
        this.setupEventListeners();
        this.loadCategories();
        this.loadBooks();
        this.loadProfileCategories();
        
        // Wait a bit longer to ensure categories are loaded
        setTimeout(() => {
            this.populateCategoryDropdowns();
        }, 1000);
    }
    
    // ?? NEW: Pagination methods for books
    getPaginatedBooks() {
        const startIndex = (this.currentBookPage - 1) * this.booksPerPage;
        const endIndex = startIndex + this.booksPerPage;
        return this.books.slice(startIndex, endIndex);
    }

    updateBookPagination() {
        this.totalBookPages = Math.ceil(this.books.length / this.booksPerPage);
        this.renderBookPagination();
    }

    renderBookPagination() {
        const container = document.getElementById('books-list');
        if (!container) {
            console.log('? Books list container not found');
            return;
        }

        // ????? ????? ???????
        const paginationHTML = this.createPaginationHTML('book', this.currentBookPage, this.totalBookPages);
        
        if (!paginationHTML) {
            console.log('?? No pagination needed for books');
            // Remove existing pagination if no longer needed
            const existingPagination = container.nextElementSibling;
            if (existingPagination && existingPagination.classList.contains('pagination-container')) {
                existingPagination.remove();
            }
            return;
        }
        
        // ????? ??????? ??? ????? ?????
        const existingPagination = container.nextElementSibling;
        if (existingPagination && existingPagination.classList.contains('pagination-container')) {
            existingPagination.remove();
        }
        
        const paginationContainer = document.createElement('div');
        paginationContainer.className = 'pagination-container';
        paginationContainer.innerHTML = paginationHTML;
        
        // ????? ??? container ??????
        container.parentNode.insertBefore(paginationContainer, container.nextSibling);
        
        console.log('? Book pagination rendered for page', this.currentBookPage, 'of', this.totalBookPages);
        
        // ????? ?????? ???????
        this.attachPaginationEventListeners('book');
    }

    // ?? NEW: Pagination methods for categories
    getPaginatedCategories() {
        const startIndex = (this.currentCategoryPage - 1) * this.categoriesPerPage;
        const endIndex = startIndex + this.categoriesPerPage;
        return this.categories.slice(startIndex, endIndex);
    }

    updateCategoryPagination() {
        this.totalCategoryPages = Math.ceil(this.categories.length / this.categoriesPerPage);
        this.renderCategoryPagination();
    }

    renderCategoryPagination() {
        const container = document.getElementById('categories-list');
        if (!container) {
            console.log('? Categories list container not found');
            return;
        }

        const paginationHTML = this.createPaginationHTML('category', this.currentCategoryPage, this.totalCategoryPages);
        
        if (!paginationHTML) {
            console.log('?? No pagination needed for categories');
            // Remove existing pagination if no longer needed
            const existingPagination = container.nextElementSibling;
            if (existingPagination && existingPagination.classList.contains('pagination-container')) {
                existingPagination.remove();
            }
            return;
        }
        
        const existingPagination = container.nextElementSibling;
        if (existingPagination && existingPagination.classList.contains('pagination-container')) {
            existingPagination.remove();
        }
        
        const paginationContainer = document.createElement('div');
        paginationContainer.className = 'pagination-container';
        paginationContainer.innerHTML = paginationHTML;
        container.parentNode.insertBefore(paginationContainer, container.nextSibling);
        
        console.log('? Category pagination rendered for page', this.currentCategoryPage, 'of', this.totalCategoryPages);
        
        this.attachPaginationEventListeners('category');
    }

    // ?? NEW: Pagination methods for profile categories
    getPaginatedProfileCategories() {
        const startIndex = (this.currentProfileCategoryPage - 1) * this.profileCategoriesPerPage;
        const endIndex = startIndex + this.profileCategoriesPerPage;
        return this.profileCategories.slice(startIndex, endIndex);
    }

    updateProfileCategoryPagination() {
        this.totalProfileCategoryPages = Math.ceil(this.profileCategories.length / this.profileCategoriesPerPage);
        this.renderProfileCategoryPagination();
    }

    renderProfileCategoryPagination() {
        const container = document.getElementById('profile-categories-list');
        if (!container) {
            console.log('? Profile categories list container not found');
            return;
        }

        const paginationHTML = this.createPaginationHTML('profile-category', this.currentProfileCategoryPage, this.totalProfileCategoryPages);
        
        if (!paginationHTML) {
            console.log('?? No pagination needed for profile categories');
            // Remove existing pagination if no longer needed
            const existingPagination = container.nextElementSibling;
            if (existingPagination && existingPagination.classList.contains('pagination-container')) {
                existingPagination.remove();
            }
            return;
        }
        
        const existingPagination = container.nextElementSibling;
        if (existingPagination && existingPagination.classList.contains('pagination-container')) {
            existingPagination.remove();
        }
        
        const paginationContainer = document.createElement('div');
        paginationContainer.className = 'pagination-container';
        paginationContainer.innerHTML = paginationHTML;
        container.parentNode.insertBefore(paginationContainer, container.nextSibling);
        
        console.log('? Profile category pagination rendered for page', this.currentProfileCategoryPage, 'of', this.totalProfileCategoryPages);
        
        this.attachPaginationEventListeners('profile-category');
    }

    // ?? NEW: Create pagination HTML
    createPaginationHTML(type, currentPage, totalPages) {
        if (totalPages <= 1) {
            return '';
        }

        let html = `
            <div class="pagination">
                <button class="pagination-btn ${currentPage === 1 ? 'disabled' : ''}" 
                        data-type="${type}" data-page="${currentPage - 1}">
                     Previous
                </button>
                <div class="pagination-numbers">
        `;

        // ??? ????? ???????
        const maxVisiblePages = 5;
        let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);

        if (endPage - startPage + 1 < maxVisiblePages) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        // ?? ?????? ??????
        if (startPage > 1) {
            html += `<button class="pagination-btn" data-type="${type}" data-page="1">1</button>`;
            if (startPage > 2) {
                html += `<span class="pagination-ellipsis">...</span>`;
            }
        }

        // ????? ???????
        for (let i = startPage; i <= endPage; i++) {
            html += `
                <button class="pagination-btn ${i === currentPage ? 'active' : ''}" 
                        data-type="${type}" data-page="${i}">
                    ${i}
                </button>
            `;
        }

        // ?? ?????? ???????
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                html += `<span class="pagination-ellipsis">...</span>`;
            }
            html += `<button class="pagination-btn" data-type="${type}" data-page="${totalPages}">${totalPages}</button>`;
        }

        html += `
                </div>
                <button class="pagination-btn ${currentPage === totalPages ? 'disabled' : ''}" 
                        data-type="${type}" data-page="${currentPage + 1}">
                    Next 
                </button>
            </div>
            <div class="pagination-info">
                Page ${currentPage} of ${totalPages} - Total ${this.getTotalItems(type)} items
            </div>
        `;

        return html;
    }

    // ?? NEW: Get total items count
    getTotalItems(type) {
        switch (type) {
            case 'book': return this.books.length;
            case 'category': return this.categories.length;
            case 'profile-category': return this.profileCategories.length;
            default: return 0;
        }
    }

    // ?? NEW: Attach pagination event listeners
    attachPaginationEventListeners(type) {
        const buttons = document.querySelectorAll(`[data-type="${type}"]`);
        console.log(`?? Attaching listeners for ${type} pagination - ${buttons.length} buttons found`);
        
        buttons.forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                if (btn.classList.contains('disabled')) {
                    console.log('?? Button disabled, ignoring click');
                    return;
                }

                const page = parseInt(btn.dataset.page);
                console.log(`?? Changing ${type} page to:`, page);
                this.changePage(type, page);
            });
        });
    }

    // ?? NEW: Change page method
    changePage(type, newPage) {
        console.log(`?? Changing ${type} page from to:`, newPage);
        
        switch (type) {
            case 'book':
                if (newPage >= 1 && newPage <= this.totalBookPages) {
                    this.currentBookPage = newPage;
                    this.renderBooks();
                    this.renderBookPagination();
                    this.scrollToTop();
                }
                break;
            case 'category':
                if (newPage >= 1 && newPage <= this.totalCategoryPages) {
                    this.currentCategoryPage = newPage;
                    this.renderCategories();
                    this.renderCategoryPagination();
                    this.scrollToTop();
                }
                break;
            case 'profile-category':
                if (newPage >= 1 && newPage <= this.totalProfileCategoryPages) {
                    this.currentProfileCategoryPage = newPage;
                    this.renderProfileCategories();
                    this.renderProfileCategoryPagination();
                    this.scrollToTop();
                }
                break;
        }
    }

    // ?? NEW: Scroll to top when changing pages
    scrollToTop() {
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }

    // ?? NEW: Reset pagination to first page
    resetPagination() {
        this.currentBookPage = 1;
        this.currentCategoryPage = 1;
        this.currentProfileCategoryPage = 1;
        console.log('?? Pagination reset to first pages');
    }
    
    // ?? NEW: ???? ?????? ???????? ??? Books
    updateBooksStats() {
        const totalBooks = this.books.length;
        const activeBooks = this.books.filter(book => 
            book.is_active === true || book.is_active === 1 || book.is_active === '1'
        ).length;

        // ????? ???????
        const totalCountElement = document.getElementById('total-books-count');
        const activeCountElement = document.getElementById('active-books-count');
        const newBooksCountElement = document.getElementById('new-books-count');
        const totalSizeElement = document.getElementById('total-size-count');
        
        if (totalCountElement) {
            totalCountElement.textContent = totalBooks;
        }
        if (activeCountElement) {
            activeCountElement.textContent = activeBooks;
        }
        
        // ?? NEW: ???? ????? ??????? (?? ??????? ?? ??? 30 ???)
        if (newBooksCountElement) {
            const thirtyDaysAgo = new Date();
            thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
            const newBooks = this.books.filter(book => {
                const createdDate = new Date(book.created_at);
                return createdDate >= thirtyDaysAgo;
            }).length;
            newBooksCountElement.textContent = newBooks;
        }
        
        // ?? NEW: ???? ????? ????????
        if (totalSizeElement) {
            const totalSizeBytes = this.books.reduce((total, book) => total + (book.file_size || 0), 0);
            const totalSizeFormatted = this.formatFileSize(totalSizeBytes);
            totalSizeElement.textContent = totalSizeFormatted;
        }

        console.log(`?? Books Stats - Total: ${totalBooks}, Active: ${activeBooks}`);
    }

    // ?? NEW: ???? ?????? ???????? ??? Categories
    updateCategoriesStats() {
        const totalCategories = this.categories.length;
        const activeCategories = this.categories.filter(cat => 
            cat.is_active === true || cat.is_active === 1 || cat.is_active === '1'
        ).length;

        // ????? ???????
        const totalCountElement = document.getElementById('total-categories-count');
        const activeCountElement = document.getElementById('active-categories-count');
        
        if (totalCountElement) {
            totalCountElement.textContent = totalCategories;
        }
        if (activeCountElement) {
            activeCountElement.textContent = activeCategories;
        }

        console.log(`?? Categories Stats - Total: ${totalCategories}, Active: ${activeCategories}`);
    }

    // ?? NEW: ???? ?????? ???????? ??? Profile Categories
    updateProfileCategoriesStats() {
        const totalProfileCategories = this.profileCategories.length;
        const activeProfileCategories = this.profileCategories.filter(cat => 
            cat.is_active === true || cat.is_active === 1 || cat.is_active === '1'
        ).length;

        // ????? ???????
        const totalCountElement = document.getElementById('total-profile-categories-count');
        const activeCountElement = document.getElementById('active-profile-categories-count');
        
        if (totalCountElement) {
            totalCountElement.textContent = totalProfileCategories;
        }
        if (activeCountElement) {
            activeCountElement.textContent = activeProfileCategories;
        }

        console.log(`?? Profile Categories Stats - Total: ${totalProfileCategories}, Active: ${activeProfileCategories}`);
    }

    // ???? ???? ??????? ???????
    async saveContactSettings() {
        try {
            console.log('Saving connection settings...');
            
            const formData = new FormData();
            formData.append('phone_number', document.getElementById('phone-number').value);
            formData.append('whatsapp_number', document.getElementById('whatsapp-number').value);
            formData.append('email', document.getElementById('email').value);
            formData.append('company_name', document.getElementById('company-name').value);
            formData.append('company_address', document.getElementById('company-address').value);
            formData.append('business_hours', document.getElementById('business-hours').value);

            const response = await fetch(`${API_BASE}/contact_settings.php`, {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Server Error');
            }

            alert('Contact settings saved successfully');
            
            this.updateContactStats();
            
        } catch (error) {
            console.error('Error saving connection settings', error);
            alert('Save error: ' + error.message);
        }
    }

    // ???? ?????? ??????????
    updateContactStats() {
        document.getElementById('phone-stat').textContent = document.getElementById('phone-number').value;
        document.getElementById('whatsapp-stat').textContent = document.getElementById('whatsapp-number').value;
        document.getElementById('email-stat').textContent = document.getElementById('email').value;
        const address = document.getElementById('company-address').value;
        document.getElementById('address-stat').textContent = address ? address.substring(0, 20) + '...' : 'No address';
    }

    // ???? ?????? ??????? ??????? ???????
    async loadContactSettings() {
        try {
            console.log('?? ???? ????? ??????? ???????...');
            
            const response = await fetch(`${API_BASE}/contact_settings.php`);
            const settings = await response.json();

            if (response.ok) {
                document.getElementById('company-name').value = settings.company_name || '';
                document.getElementById('phone-number').value = settings.phone_number || '';
                document.getElementById('whatsapp-number').value = settings.whatsapp_number || '';
                document.getElementById('email').value = settings.email || '';
                document.getElementById('company-address').value = settings.company_address || '';
                document.getElementById('business-hours').value = settings.business_hours || '';
                
                this.updateContactStats();
            }
        } catch (error) {
            console.error('? ??? ?? ????? ??????? ???????:', error);
        }
    }

    // ?? ????? ???? loadScreenTexts ?? script.js
async loadScreenTexts() {
    try {
        console.log('?? ???? ????? ???? ???????...');
        
        const response = await fetch(`${API_BASE}/screen_texts.php`);
        const texts = await response.json();

        if (response.ok) {
            // ?????? ?????? ????????
            texts.forEach(text => {
                if (text.screen_name === 'home_screen') {
                    document.getElementById('home-screen-title').value = text.title_text || '';
                    document.getElementById('home-screen-description').value = text.description_text || '';
                    
                    // ?????? ??????? ??????: ?????? ???? show_welcome_section ???? ????
                    const showWelcome = text.show_welcome_section === true || 
                                      text.show_welcome_section === 1 || 
                                      text.show_welcome_section === '1' ||
                                      text.show_welcome_section === null; // ??? ???? null ??????? true
                    
                    document.getElementById('home-show-welcome-section').checked = showWelcome;
                    document.getElementById('home-text-stat').textContent = text.title_text ? ' Configured' : 'Not Set';
                    
                } else if (text.screen_name === 'profile_screen') {
                    document.getElementById('profile-screen-title').value = text.title_text || '';
                    document.getElementById('profile-screen-description').value = text.description_text || '';
                    
                    // ?????? ??????? ??????: ?????? ???? show_welcome_section ???? ????
                    const showProfileWelcome = text.show_welcome_section === true || 
                                             text.show_welcome_section === 1 || 
                                             text.show_welcome_section === '1' ||
                                             text.show_welcome_section === null; // ??? ???? null ??????? true
                    
                    document.getElementById('profile-show-welcome-section').checked = showProfileWelcome;
                    document.getElementById('profile-text-stat').textContent = text.title_text ? ' Configured' : 'Not Set';
                }
            });
            
            console.log('? ?? ????? ???? ??????? ?????');
        }
    } catch (error) {
        console.error('? ??? ?? ????? ???? ???????:', error);
    }
}

// ?? ???? ?? ??? ?????? ?? script.js
async saveScreenTexts() {
    try {
        console.log('?? ???? ??? ???? ???????...');
        
        // ?? DEBUG: ????? ?????? ???????
        const homeShowWelcome = document.getElementById('home-show-welcome-section').checked;
        const profileShowWelcome = document.getElementById('profile-show-welcome-section').checked;
        
        console.log('?? DEBUG - Checkbox values:');
        console.log('   Home Show Welcome:', homeShowWelcome);
        console.log('   Profile Show Welcome:', profileShowWelcome);
        
        const textsToSave = [
            {
                screen_name: 'home_screen',
                title_text: document.getElementById('home-screen-title').value,
                description_text: document.getElementById('home-screen-description').value,
                show_welcome_section: homeShowWelcome
            },
            {
                screen_name: 'profile_screen',
                title_text: document.getElementById('profile-screen-title').value,
                description_text: document.getElementById('profile-screen-description').value,
                show_welcome_section: profileShowWelcome
            }
        ];

        console.log('?? DEBUG - Data being sent:', JSON.stringify(textsToSave, null, 2));
        
        let savedCount = 0;
        let errors = [];
        
        for (const text of textsToSave) {
            try {
                const response = await fetch(`${API_BASE}/screen_texts.php`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(text)
                });

                const result = await response.json();
                console.log(`?? Response for ${text.screen_name}:`, result);

                if (response.ok) {
                    savedCount++;
                    console.log(`? ?? ??? ?? ${text.screen_name} - show_welcome_section: ${text.show_welcome_section}`);
                } else {
                    errors.push(`${text.screen_name}: ${result.error || 'Unknown error'}`);
                    console.error(`? ??? ?? ??? ?? ${text.screen_name}:`, result.error);
                }
            } catch (error) {
                errors.push(`${text.screen_name}: ${error.message}`);
                console.error(`? ??? ?? ??? ?? ${text.screen_name}:`, error);
            }
        }

        if (savedCount === textsToSave.length) {
            alert(' All texts have been saved successfully!');
            this.updateScreenTextStats();
        } else {
            alert(`?? Only ${savedCount} out of ${textsToSave.length} texts saved. Errors: ${errors.join(', ')}`);
        }
        
    } catch (error) {
        console.error('? Error saving screen text:', error);
        alert('? Save error: ' + error.message);
    }
}

    // ?? ????? ???? updateScreenTextStats
updateScreenTextStats() {
    const homeTitle = document.getElementById('home-screen-title').value;
    const profileTitle = document.getElementById('profile-screen-title').value;
    const showWelcome = document.getElementById('home-show-welcome-section').checked;
    const showProfileWelcome = document.getElementById('profile-show-welcome-section').checked;
    
    document.getElementById('home-text-stat').textContent = homeTitle ? 
        (showWelcome ? ' Configured (Visible)' : ' Configured (Hidden)') : 'Not Set';
    document.getElementById('profile-text-stat').textContent = profileTitle ? 
        (showProfileWelcome ? ' Configured (Visible)' : ' Configured (Hidden)') : 'Not Set';
}

    setupEventListeners() {
		
		// ?? NEW: Bulk delete event listener
document.getElementById('delete-selected-books')?.addEventListener('click', () => {
    this.deleteSelectedBooks();
});
		
		// ?? NEW: Global Auth Control listeners
document.getElementById('save-global-auth')?.addEventListener('click', () => {
    this.updateGlobalAuthSettings();
});

// ?? NEW: Navigation to global auth section
document.querySelector('[data-section="global-auth"]')?.addEventListener('click', () => {
    setTimeout(() => {
        this.loadGlobalAuthSection();
    }, 100);
});
        

    console.log('Setting up event listeners...');
    
    // ??? ?????? ????????? ?? ???? ???????
    document.querySelectorAll('form').forEach(form => {
        form.setAttribute('novalidate', 'true');
    });
    
    // Taille Extrait toggle listener
    document.getElementById('book-show-taille-before-extrait')?.addEventListener('change', (e) => {
        this.toggleTailleExtraitGroup(e.target.checked);
    });
    
    // ?? NEW: External Download Link toggle listener
    document.getElementById('book-show-external-download-button')?.addEventListener('change', (e) => {
        this.toggleExternalDownloadLinkGroup(e.target.checked);
    });
    
    // ?????? **FIXED: External Download Protection toggle listener**
    document.getElementById('book-is-external-download-protected')?.addEventListener('change', (e) => {
        console.log('?? External Download Protection checkbox changed:', e.target.checked);
        this.toggleExternalDownloadProtectionGroup(e.target.checked);
    });
    
    // ?? NEW: ????? ??? ?????? ??????? - ????
    document.getElementById('book-protection-enabled')?.addEventListener('change', (e) => {
        const keyGroup = document.getElementById('protection-key-group');
        if (keyGroup) {
            keyGroup.style.display = e.target.checked ? 'block' : 'none';
            
            // ??? ?? ????? ???????? ???? ??? ???????
            if (!e.target.checked) {
                document.getElementById('book-protection-key').value = '';
            }
            
            console.log('?? Protection toggled:', e.target.checked);
        }
    });
    
        
        // ????? ??? ???? ??????? ???????
        document.getElementById('save-contact-settings')?.addEventListener('click', () => {
            this.saveContactSettings();
        });

        // ????? ??? ???? ???? ???????
        document.getElementById('save-screen-texts')?.addEventListener('click', () => {
            this.saveScreenTexts();
        });

        // ????? ????????? ??? ??? ??? ???????
        document.querySelector('[data-section="contact-settings"]')?.addEventListener('click', () => {
            setTimeout(() => {
                this.loadContactSettings();
            }, 100);
        });

        // ????? ?????? ??? ??? ??? ??????
        document.querySelector('[data-section="screen-texts"]')?.addEventListener('click', () => {
            setTimeout(() => {
                this.loadScreenTexts();
            }, 100);
        });

        // ????? ?????????? ??? ????? ??????
        ['home-screen-title', 'profile-screen-title'].forEach(id => {
            document.getElementById(id)?.addEventListener('input', () => {
                this.updateScreenTextStats();
            });
        });
        
        // Navigation between sections
        document.querySelectorAll('.nav-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const section = e.target.dataset.section;
                console.log('Clicking on section:', section);
                this.showSection(section);
            });
        });

        // Open modals
        document.getElementById('add-category-btn')?.addEventListener('click', () => {
            this.openCategoryModal();
        });

        document.getElementById('add-book-btn')?.addEventListener('click', () => {
            this.openBookModal();
        });

        document.getElementById('add-profile-category-btn')?.addEventListener('click', () => {
            this.openProfileCategoryModal();
        });

        // Close modals
        document.querySelectorAll('.close, .btn-secondary').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                this.closeModals();
            });
        });

        // Submit forms
        document.getElementById('category-form')?.addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveCategory();
        });

        document.getElementById('book-form')?.addEventListener('submit', (e) => {
            console.log('?? Book form submit event triggered');
            e.preventDefault();
            this.saveBook(e);
        });

        document.getElementById('profile-category-form')?.addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveProfileCategory();
        });

        // Image previews
        this.setupImagePreviews();

        // Close modals when clicking outside
        window.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal')) {
                this.closeModals();
            }
        });

        // Page range validation
        const summaryPagesInput = document.getElementById('book-selected-pages-summary');
        const tocPagesInput = document.getElementById('book-selected-pages-toc');
        const originalPagesInput = document.getElementById('book-selected-pages-original');
        
        [summaryPagesInput, tocPagesInput, originalPagesInput].forEach(input => {
            if (input) {
                input.addEventListener('input', (e) => {
                    this.validatePageRanges(e.target);
                });
            }
        });

        // PDF source toggle
        this.setupPdfSourceToggle();

        // Delete cover image buttons
        this.setupDeleteImageButtons();

        console.log('All listeners set up successfully');
    }

    setupImagePreviews() {
        const imageInputs = [
            { input: 'category-image', preview: 'category-image-preview' },
            { input: 'book-image', preview: 'book-image-preview' },
            { input: 'profile-category-image', preview: 'profile-category-image-preview' }
        ];

        imageInputs.forEach(({ input, preview }) => {
            const inputElement = document.getElementById(input);
            if (inputElement) {
                inputElement.addEventListener('change', (e) => {
                    this.previewImage(e.target, preview);
                    inputElement.removeAttribute('data-remove-image');
                });
            }
        });
    }

    setupDeleteImageButtons() {
        // Delete cover image for book
        const deleteBookImageBtn = document.getElementById('delete-book-image');
        if (deleteBookImageBtn) {
            deleteBookImageBtn.addEventListener('click', () => {
                this.deleteBookImage();
            });
        }

        // Delete image for category
        const deleteCategoryImageBtn = document.getElementById('delete-category-image');
        if (deleteCategoryImageBtn) {
            deleteCategoryImageBtn.addEventListener('click', () => {
                this.deleteCategoryImage();
            });
        }

        // Delete image for profile category
        const deleteProfileCategoryImageBtn = document.getElementById('delete-profile-category-image');
        if (deleteProfileCategoryImageBtn) {
            deleteProfileCategoryImageBtn.addEventListener('click', () => {
                this.deleteProfileCategoryImage();
            });
        }
    }

    setupPdfSourceToggle() {
        const pdfOptionRadios = document.querySelectorAll('input[name="pdf_url_option"]');
        const fileUploadGroup = document.getElementById('file-upload-group');
        const externalUrlGroup = document.getElementById('external-url-group');
        const externalUrlPlusGroup = document.getElementById('external-url-plus-group');
        const pdfFileInput = document.getElementById('book-pdf');
        const externalUrlInput = document.getElementById('external-pdf-url');
        const externalUrlPlusInput = document.getElementById('external-pdf-url-plus');

        pdfOptionRadios.forEach(radio => {
            radio.addEventListener('change', (e) => {
                fileUploadGroup.style.display = 'none';
                externalUrlGroup.style.display = 'none';
                externalUrlPlusGroup.style.display = 'none';
                
                pdfFileInput.removeAttribute('required');
                externalUrlInput.removeAttribute('required');
                externalUrlPlusInput.removeAttribute('required');

                if (e.target.value === 'url') {
                    externalUrlGroup.style.display = 'block';
                } else if (e.target.value === 'url_plus') {
                    externalUrlPlusGroup.style.display = 'block';
                } else {
                    fileUploadGroup.style.display = 'block';
                }
            });
        });
    }

    validatePageRanges(input) {
        const value = input.value.trim();
        if (!value) {
            input.style.borderColor = '#e2e8f0';
            input.title = '';
            return true;
        }

        const ranges = value.split(',');
        let isValid = true;
        
        for (let range of ranges) {
            range = range.trim();
            if (range.includes('-')) {
                const [start, end] = range.split('-').map(num => parseInt(num.trim()));
                if (isNaN(start) || isNaN(end) || start <= 0 || end <= 0 || start > end) {
                    isValid = false;
                    break;
                }
            } else {
                const page = parseInt(range);
                if (isNaN(page) || page <= 0) {
                    isValid = false;
                    break;
                }
            }
        }

        if (!isValid) {
            input.style.borderColor = '#e53e3e';
            input.title = 'Format invalide. Utilisez: 8-10 ou 1-5,10-15,20';
        } else {
            input.style.borderColor = '#e2e8f0';
            input.title = '';
        }
        
        return isValid;
    }

    previewImage(input, previewId) {
        const preview = document.getElementById(previewId);
        if (!preview) return;
        
        preview.innerHTML = '';
        
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            
            reader.onload = function(e) {
                const img = document.createElement('img');
                img.src = e.target.result;
                preview.appendChild(img);

                const deleteBtn = input.parentElement.querySelector('.delete-image-btn');
                if (deleteBtn) {
                    deleteBtn.style.display = 'inline-block';
                }
            }
            
            reader.readAsDataURL(input.files[0]);
        }
    }

    deleteBookImage() {
        const imageInput = document.getElementById('book-image');
        const preview = document.getElementById('book-image-preview');
        const deleteBtn = document.getElementById('delete-book-image');
        
        imageInput.value = '';
        if (preview) preview.innerHTML = '';
        if (deleteBtn) deleteBtn.style.display = 'none';
        
        // ?????? **??????? ??????: ?? ??? data-remove-image ?????? ??????????**
        // ??? ??? ???? ???? ???? ?????? ?????? ????? ?????? ??????????
        const currentImage = document.querySelector('#book-image-preview img');
        if (currentImage && !currentImage.src.includes('no-image.jpg')) {
            imageInput.setAttribute('data-remove-image', 'true');
            console.log('Book custom image marked for deletion');
        } else {
            imageInput.removeAttribute('data-remove-image');
            console.log('Default book image - not marking for deletion');
        }
    }

    deleteCategoryImage() {
        const imageInput = document.getElementById('category-image');
        const preview = document.getElementById('category-image-preview');
        const deleteBtn = document.getElementById('delete-category-image');
        
        imageInput.value = '';
        if (preview) preview.innerHTML = '';
        if (deleteBtn) deleteBtn.style.display = 'none';
        
        // ?????? **??????? ??????: ?? ??? data-remove-image ?????? ??????????**
        const currentImage = document.querySelector('#category-image-preview img');
        if (currentImage && !currentImage.src.includes('no-image')) {
            imageInput.setAttribute('data-remove-image', 'true');
            console.log('Category custom image marked for deletion');
        } else {
            imageInput.removeAttribute('data-remove-image');
            console.log('Default category image - not marking for deletion');
        }
    }

    deleteProfileCategoryImage() {
        const imageInput = document.getElementById('profile-category-image');
        const preview = document.getElementById('profile-category-image-preview');
        const deleteBtn = document.getElementById('delete-profile-category-image');
        
        imageInput.value = '';
        if (preview) preview.innerHTML = '';
        if (deleteBtn) deleteBtn.style.display = 'none';
        
        // ?????? **??????? ??????: ?? ??? data-remove-image ?????? ??????????**
        const currentImage = document.querySelector('#profile-category-image-preview img');
        if (currentImage && !currentImage.src.includes('no-image')) {
            imageInput.setAttribute('data-remove-image', 'true');
            console.log('Profile category custom image marked for deletion');
        } else {
            imageInput.removeAttribute('data-remove-image');
            console.log('Default profile category image - not marking for deletion');
        }
    }

    // ?? UPDATE: ????? ???? showSection ?????? ???? ??????????
showSection(sectionName) {
    console.log('Showing section:', sectionName);
    
    document.querySelectorAll('.nav-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    const activeBtn = document.querySelector(`[data-section="${sectionName}"]`);
    if (activeBtn) {
        activeBtn.classList.add('active');
    }

    document.querySelectorAll('.section').forEach(section => {
        section.classList.remove('active');
    });
    
    const targetSection = document.getElementById(`${sectionName}-section`);
    if (targetSection) {
        targetSection.classList.add('active');
    }
    
    // ?? NEW: ????? ?????????? ??? ??? ??????? ????????
    setTimeout(() => {
        switch (sectionName) {
            case 'categories':
                this.updateCategoriesStats();
                break;
            case 'books':
                this.updateBooksStats();
                break;
            case 'profile-categories':
                this.updateProfileCategoriesStats();
                break;
            case 'contact-settings':
                this.updateContactStats();
                break;
            case 'screen-texts':
                this.updateScreenTextStats();
                break;
            case 'global-auth':  // <<=== ADDED: Global Auth Section
                this.loadGlobalAuthSection();
                break;
        }
    }, 100);
    
    // ?? NEW: Reset to first page when switching sections
    this.resetPagination();
}

    // ?? UPDATE: ????? ???? loadCategories ?????? ??????????
    async loadCategories() {
        try {
            console.log('Loading categories...');
            const response = await fetch(`${API_BASE}/categories.php`);
            
            if (!response.ok) {
                throw new Error(`Server error: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.categories = Array.isArray(data) ? data : [];
            this.renderCategories();
            
            // ?? NEW: ????? ?????????? ??? ???????
            this.updateCategoriesStats();
            
        } catch (error) {
            console.error('Error loading categories:', error);
            this.showError('categories-list', 'Error loading categories: ' + error.message);
            
            // ?? NEW: ????? ?????????? ??? ?? ???? ?????
            this.updateCategoriesStats();
        }
    }

    // ?? UPDATE: ????? ???? loadBooks ?????? ??????????
async loadBooks() {
    try {
        console.log('Loading books...');
        const response = await fetch(`${API_BASE}/books.php`);
        
        if (!response.ok) {
            throw new Error(`Server error: ${response.status}`);
        }
        
        const data = await response.json();
        
        if (data.error) {
            throw new Error(data.error);
        }
        
        this.books = Array.isArray(data) ? data : [];
        
        // ?? FIX: ?????? ?? ?? ???????? ????? ??? ???????
        if (this.books.length > 0) {
            console.log(`?? Loaded ${this.books.length} books successfully`);
            this.renderBooks();
        } else {
            console.log('?? No books found');
            this.renderBooks(); // ?? FIX: ??????? renderBooks ??? ?? ?? ???? ???
        }
        
        // ?? NEW: ????? ?????????? ??? ???????
        this.updateBooksStats();
        
    } catch (error) {
        console.error('Error loading books:', error);
        this.showError('books-list', 'Error loading books: ' + error.message);
        
        // ?? FIX: ????? ????? ??????? ????? ?? ???? ?????
        this.books = [];
        this.renderBooks();
        
        // ?? NEW: ????? ?????????? ??? ?? ???? ?????
        this.updateBooksStats();
    }
}

    // ?? UPDATE: ????? ???? loadProfileCategories ?????? ??????????
    async loadProfileCategories() {
        try {
            console.log('Loading profile categories...');
            const response = await fetch(`${API_BASE}/profile_categories.php`);
            
            if (!response.ok) {
                if (response.status === 404) {
                    console.log('Profile categories file does not exist yet');
                    this.profileCategories = [];
                    
                    // ?? ????? ?????????? ??? ?? ??? ????? ??? ?????
                    this.updateProfileCategoriesStats();
                    return;
                }
                throw new Error(`Server error: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.profileCategories = Array.isArray(data) ? data : [];
            this.renderProfileCategories();
            
            // ?? NEW: ????? ?????????? ??? ???????
            this.updateProfileCategoriesStats();
            
        } catch (error) {
            console.error('Error loading profile categories:', error);
            this.profileCategories = [];
            
            // ?? NEW: ????? ?????????? ??? ?? ???? ?????
            this.updateProfileCategoriesStats();
        }
    }

    populateCategoryDropdowns() {
        const regularDropdown = document.getElementById('book-category');
        const profileDropdown = document.getElementById('book-profile-category');
        
        console.log('Populating category dropdowns...');
        console.log('Regular categories:', this.categories);
        console.log('Profile categories:', this.profileCategories);
        
        if (regularDropdown) {
            regularDropdown.innerHTML = '<option value="">Select a regular category</option>' +
                this.categories.map(cat => 
                    `<option value="${cat.id}">${cat.name}</option>`
                ).join('');
            console.log('Regular dropdown populated with', this.categories.length, 'categories');
        }
        
        if (profileDropdown) {
            profileDropdown.innerHTML = '<option value="">Select a research category</option>' +
                this.profileCategories.map(cat => 
                    `<option value="${cat.id}">${cat.name}</option>`
                ).join('');
            console.log('Profile dropdown populated with', this.profileCategories.length, 'categories');
        }
    }

    showError(containerId, message) {
        const container = document.getElementById(containerId);
        if (container) {
            container.innerHTML = `<div class="error">${message}</div>`;
        }
    }

    // ?? UPDATE: ????? ???? renderCategories ?????? ??????????
    renderCategories() {
        const container = document.getElementById('categories-list');
        if (!container) return;
        
        const paginatedCategories = this.getPaginatedCategories();
        
        if (this.categories.length === 0) {
            container.innerHTML = '<div class="loading">No categories found</div>';
            const existingPagination = container.nextElementSibling;
            if (existingPagination && existingPagination.classList.contains('pagination-container')) {
                existingPagination.remove();
            }
            
            // ?? ????? ?????????? ??? ?? ???? ??????? ?????
            this.updateCategoriesStats();
            return;
        }
        
        console.log(`?? Rendering ${paginatedCategories.length} categories out of ${this.categories.length} total`);
        
        container.innerHTML = paginatedCategories.map(category => {
            let imageUrl = category.image_url || 'https://via.placeholder.com/80x80?text=No+Image';
            
            if (imageUrl.startsWith('uploads/')) {
                imageUrl = '../api/' + imageUrl;
            }
            
            const isActive = Boolean(category.is_active && category.is_active !== 0 && category.is_active !== '0');
            
            const statusBadge = isActive ? 
                '<span class="status-indicator active">Active</span>' :
                '<span class="status-indicator inactive">Inactive</span>';
            
            return `
                <div class="book-list-item">
                    <img src="${imageUrl}" 
                         alt="${category.name}" 
                         class="book-list-cover"
                         onerror="this.src='https://via.placeholder.com/80x80?text=No+Image'">
                    
                    <div class="book-list-content">
                        <div class="book-list-header">
                            <h3 class="book-list-title">${category.name}</h3>
                        </div>
                        
                        <div class="book-list-status">
                            ${statusBadge}
                            <span class="status-indicator">Order: ${category.display_order || 0}</span>
                        </div>
                        
                        <div class="book-list-actions">
                            <button class="btn-primary" onclick="adminApp.editCategory(${category.id})">Edit</button>
                            <button class="btn-secondary" onclick="adminApp.deleteCategory(${category.id})">Delete</button>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        this.updateCategoryPagination();
        
        // ?? NEW: ????? ?????????? ??? ???????
        this.updateCategoriesStats();
    }

    // ?? UPDATE: ????? ???? renderBooks ?????? ??????????
renderBooks() {
    const container = document.getElementById('books-list');
    if (!container) return;
    
    // ?? UPDATE: Use paginated data - ????? ??????? ?????
    const paginatedBooks = this.getPaginatedBooks();
    
    if (this.books.length === 0) {
        container.innerHTML = '<div class="loading">No books found</div>';
        // Remove any existing pagination
        const existingPagination = container.nextElementSibling;
        if (existingPagination && existingPagination.classList.contains('pagination-container')) {
            existingPagination.remove();
        }
        
        // ?? ????? ?????????? ??? ?? ???? ??????? ?????
        this.updateBooksStats();
        return;
    }
    
    console.log(`?? Rendering ${paginatedBooks.length} books out of ${this.books.length} total`);
    
    // ?? NEW: Add select all functionality (optional)
    const selectAllHTML = `
        <div class="selection-header">
            
            <div class="select-all-container">
                <input type="checkbox" id="select-all-books" class="select-all-checkbox">
                <label for="select-all-books">Select All</label>
            </div>
        </div>
    `;
    
    // ????? ??? HTML ??? ?? ?????? ??? ???? ???????? ????????
    let booksHTML = paginatedBooks.map(book => {
        let imageUrl = book.image_url || 'uploads/no-image.jpg';
        
        if (imageUrl.startsWith('uploads/')) {
            imageUrl = '../api/' + imageUrl;
        }

        // ?? NEW: Active status indicator - FIXED
        const activeIndicator = book.is_active ? 
            '<span class="status-indicator active">Active</span>' :
            '<span class="status-indicator inactive">Hidden</span>';
            
        // ?? NEW: Login Required status indicator
        const isLoginRequired = (
            book.require_login_for_download === true || 
            book.require_login_for_download === 1 || 
            book.require_login_for_download === '1' ||
            String(book.require_login_for_download).toLowerCase() === 'true'
        );

        const loginRequiredIndicator = isLoginRequired ? 
            '<span class="status-indicator login-required">Login Required</span>' : '';

        // ????? ??????? ?????? - ??? ???????? ???????
        const externalPlusIndicator = book.is_external_plus ? 
            '<span class="status-indicator external-plus">External Link Plus</span>' : '';
            
        const externalLinkIndicator = book.external_download_link && book.show_external_download_button ? 
            '<span class="status-indicator external">Download Link</span>' : '';
            
        const externalProtectionIndicator = book.is_external_download_protected ? 
            '<span class="status-indicator protected">Protected Link</span>' : '';
            
        const formatIndicator = book.format ? 
            `<span class="status-indicator format">${book.format}</span>` : '';
            
        const externalIndicator = book.is_external ? 
            '<span class="status-indicator external">External Link</span>' : '';

        const summaryPdfIndicator = book.extracted_summary_pdf_url ? 
            '<span class="status-indicator">Summary PDF</span>' : '';

        const tocPdfIndicator = book.extracted_pdf_url ? 
            '<span class="status-indicator">TOC PDF</span>' : '';

        const originalPdfIndicator = book.selected_pages_original ? 
            `<span class="status-indicator">L'extrait</span>` : '';

        const protectionIndicator = book.is_protected ? 
            '<span class="status-indicator protected">Protected</span>' : '';

        const fileSizeControlIndicator = !book.show_file_size ? 
            '<span class="status-indicator size-hidden">?? Size Hidden</span>' : '';

        const tailleBeforeExtraitIndicator = book.taille_before_extrait ? 
            `<span class="status-indicator taille">?? Taille</span>` : '';

        const showTailleBeforeExtraitIndicator = !book.show_taille_before_extrait ? 
            '<span class="status-indicator size-hidden">??? Hidden</span>' : '';

        return `
            <div class="book-list-item" data-book-id="${book.id}">
                <!-- ?? NEW: Checkbox for selection -->
                <div class="book-select-checkbox">
                    <input type="checkbox" class="book-checkbox" data-book-id="${book.id}">
                </div>
                
                <img src="${imageUrl}" 
                     alt="${book.title}" 
                     class="book-list-cover"
                     onerror="this.src='https://via.placeholder.com/80x100?text=No+Image'">
                
                <div class="book-list-content">
                    <div class="book-list-header">
                        <h3 class="book-list-title">${book.title}</h3>
                    </div>
                    
                    <div class="book-list-status">
                        ${activeIndicator} <!-- ?? NEW: Active status display -->
                        ${loginRequiredIndicator} <!-- ?? NEW: Login Required status -->
                        ${originalPdfIndicator} 
                        ${protectionIndicator}
                        ${externalProtectionIndicator} 
                        ${externalLinkIndicator}
                        
                    </div>
                    
                    <div class="book-list-actions">
                        <button class="btn-primary" onclick="adminApp.editBook(${book.id})">Edit</button>
                        <button class="btn-secondary" onclick="adminApp.deleteBook(${book.id})">Delete</button>
                    </div>
                </div>
            </div>
        `;
    }).join('');
    
    // ?? NEW: Combine select all header with books
    container.innerHTML = selectAllHTML + booksHTML;
    
    // ?? UPDATE: Update pagination - ??? ??????? ???????
    this.updateBookPagination();
    
    // ?? NEW: ????? ?????????? ??? ???????
    this.updateBooksStats();
    
    // ?? NEW: ????? ??????? ??????? ???????
    this.setupBookSelection();
    
    // ?? NEW: Setup select all functionality
    const selectAllCheckbox = document.getElementById('select-all-books');
    if (selectAllCheckbox) {
        selectAllCheckbox.addEventListener('change', (e) => {
            const checkboxes = document.querySelectorAll('.book-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.checked = e.target.checked;
            });
            this.updateDeleteButtonState();
        });
    }
}

// ?? NEW: ???? ?????? ?????? ???????
setupBookSelection() {
    const checkboxes = document.querySelectorAll('.book-checkbox');
    const deleteSelectedBtn = document.getElementById('delete-selected-books');
    
    // ?? FIX: ????? ?? ??????? ????? ????? ???? ???????
    checkboxes.forEach(checkbox => {
        // ?? FIX: ????? ????????? ??????? ?????
        checkbox.replaceWith(checkbox.cloneNode(true));
    });
    
    // ?? FIX: ????? ?????? ??? ??????? ??? ?????????
    const freshCheckboxes = document.querySelectorAll('.book-checkbox');
    
    // ????? ?????? ??? ??? ???????
    freshCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', () => {
            this.updateDeleteButtonState();
        });
    });
    
    // ?? FIX: ????? ?? ??????? ????? ??? ?? ?????
    if (deleteSelectedBtn) {
        // ?? FIX: ??????? ???? ?????? ????????? ???????
        const newDeleteBtn = deleteSelectedBtn.cloneNode(true);
        deleteSelectedBtn.parentNode.replaceChild(newDeleteBtn, deleteSelectedBtn);
        
        // ?? FIX: ????? ????? ??? ????
        newDeleteBtn.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            this.deleteSelectedBooks();
        });
    }
}

// ?? NEW: ???? ?????? ?? ??? ????? ???????
updateDeleteButtonState() {
    const deleteSelectedBtn = document.getElementById('delete-selected-books');
    const selectedBooks = document.querySelectorAll('.book-checkbox:checked');
    
    if (deleteSelectedBtn) {
        if (selectedBooks.length > 0) {
            deleteSelectedBtn.style.display = 'inline-block';
            deleteSelectedBtn.innerHTML = `<i class="icon"></i> Delete Selected (${selectedBooks.length})`;
        } else {
            deleteSelectedBtn.style.display = 'none';
        }
    }
}

// ?? NEW: ???? ????? ?????? ???????
async deleteSelectedBooks() {
    const selectedCheckboxes = document.querySelectorAll('.book-checkbox:checked');
    
    if (selectedCheckboxes.length === 0) {
        alert('Please select at least one book to delete');
        return;
    }
    
    const selectedIds = Array.from(selectedCheckboxes).map(checkbox => 
        parseInt(checkbox.getAttribute('data-book-id'))
    );
    
    const bookTitles = selectedIds.map(id => {
        const book = this.books.find(b => b.id === id);
        return book ? `"${book.title}"` : `Book #${id}`;
    });
    
    // ?? FIX: ??? ????? ????? ????? ???
    const confirmationMessage = `Are you sure you want to delete the following ${selectedIds.length} book(s)? This action cannot be undone!`;
    
    // ?? FIX: ??????? confirm ??? ????? ???
    const userConfirmed = confirm(confirmationMessage);
    
    if (!userConfirmed) {
        console.log('?? User cancelled bulk delete');
        return;
    }
    
    try {
        console.log(`?? Deleting ${selectedIds.length} selected books:`, selectedIds);
        
        // ?? FIX: ????? ?? ????? ?????
        const deleteSelectedBtn = document.getElementById('delete-selected-books');
        if (deleteSelectedBtn) {
            deleteSelectedBtn.style.display = 'none';
        }
        
        // ?? FIX: ????? ???? ?????
        const container = document.getElementById('books-list');
        const originalContent = container.innerHTML;
        container.innerHTML = '<div class="loading">Deleting selected books...</div>';
        
        let successCount = 0;
        let errorCount = 0;
        const errors = [];
        
        // ????? ?? ??? ????? ??????? (????? ?????? ?? ???? ????)
        for (const bookId of selectedIds) {
            try {
                const response = await fetch(`${API_BASE}/books.php`, {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `id=${bookId}`
                });

                const result = await response.json();

                if (response.ok) {
                    successCount++;
                    console.log(`? Book ${bookId} deleted successfully`);
                } else {
                    errorCount++;
                    errors.push(`Book ${bookId}: ${result.error || 'Unknown error'}`);
                    console.error(`? Failed to delete book ${bookId}:`, result.error);
                }
            } catch (error) {
                errorCount++;
                errors.push(`Book ${bookId}: ${error.message}`);
                console.error(`? Error deleting book ${bookId}:`, error);
            }
        }
        
        // ?????? ???????
        let message = `Successfully deleted ${successCount} book(s).`;
        if (errorCount > 0) {
            message += `\n\nFailed to delete ${errorCount} book(s).\n\nErrors:\n${errors.join('\n')}`;
        }
        
        // ?? FIX: ??? ????? ??????? ??? ????? ???
        alert(message);
        
        // ?????? ????? ???????
        await this.loadBooks();
        
        // ?? FIX: ????? ????? ???? ????? ????????
        const allCheckboxes = document.querySelectorAll('.book-checkbox');
        allCheckboxes.forEach(checkbox => {
            checkbox.checked = false;
        });
        
        // ?? FIX: ????? ????? Select All
        const selectAllCheckbox = document.getElementById('select-all-books');
        if (selectAllCheckbox) {
            selectAllCheckbox.checked = false;
        }
        
    } catch (error) {
        console.error('? Error in bulk delete:', error);
        alert('Error during bulk delete: ' + error.message);
        
        // ?? FIX: ????? ????? ????? ?? ???? ?????
        await this.loadBooks();
    }
}

    // ?? UPDATE: ????? ???? renderProfileCategories ?????? ??????????
    renderProfileCategories() {
        const container = document.getElementById('profile-categories-list');
        if (!container) return;
        
        const paginatedProfileCategories = this.getPaginatedProfileCategories();
        
        if (this.profileCategories.length === 0) {
            container.innerHTML = '<div class="loading">No profile categories found</div>';
            const existingPagination = container.nextElementSibling;
            if (existingPagination && existingPagination.classList.contains('pagination-container')) {
                existingPagination.remove();
            }
            
            // ?? ????? ?????????? ??? ?? ???? ??????? ?????
            this.updateProfileCategoriesStats();
            return;
        }
        
        console.log(`?? Rendering ${paginatedProfileCategories.length} profile categories out of ${this.profileCategories.length} total`);
        
        container.innerHTML = paginatedProfileCategories.map(category => {
            let imageUrl = category.image_url || 'https://via.placeholder.com/80x80?text=No+Image';
            
            if (imageUrl.startsWith('uploads/')) {
                imageUrl = '../api/' + imageUrl;
            }
            
            const isActive = Boolean(category.is_active && category.is_active !== 0 && category.is_active !== '0');
            
            const statusBadge = isActive ? 
                '<span class="status-indicator active">Active</span>' :
                '<span class="status-indicator inactive">Inactive</span>';
            
            return `
                <div class="book-list-item">
                    <img src="${imageUrl}" 
                         alt="${category.name}" 
                         class="book-list-cover"
                         onerror="this.src='https://via.placeholder.com/80x80?text=No+Image'">
                    
                    <div class="book-list-content">
                        <div class="book-list-header">
                            <h3 class="book-list-title">${category.name}</h3>
                        </div>
                        
                        <div class="book-list-status">
                            ${statusBadge}
                            <span class="status-indicator">Order: ${category.display_order || 0}</span>
                        </div>
                        
                        <div class="book-list-actions">
                            <button class="btn-primary" onclick="adminApp.editProfileCategory(${category.id})">Edit</button>
                            <button class="btn-secondary" onclick="adminApp.deleteProfileCategory(${category.id})">Delete</button>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        this.updateProfileCategoryPagination();
        
        // ?? NEW: ????? ?????????? ??? ???????
        this.updateProfileCategoriesStats();
    }

    openCategoryModal(category = null) {
        const modal = document.getElementById('category-modal');
        const title = document.getElementById('category-modal-title');
        const form = document.getElementById('category-form');
        
        if (!modal || !title || !form) return;
        
        form.reset();
        document.getElementById('category-image').removeAttribute('data-remove-image');
        
        if (category) {
            title.textContent = 'Edit Category';
            document.getElementById('category-id').value = category.id;
            document.getElementById('category-name').value = category.name;
            document.getElementById('category-description').value = category.description || '';
            document.getElementById('category-order').value = category.display_order || 0;
            
            // ?????? ??????? ???????: ????? ???? ??? checkbox ????? ??? ?????? ???????? ?? ????? ????????
            const isActiveCheckbox = document.getElementById('category-active');
            if (isActiveCheckbox) {
                // ????? ?????? ??? boolean ???? ????
                const isActive = Boolean(category.is_active && category.is_active !== 0 && category.is_active !== '0');
                isActiveCheckbox.checked = isActive;
                console.log('?? Setting category active status:', isActive, 'from database value:', category.is_active, 'type:', typeof category.is_active);
            }
            
            const preview = document.getElementById('category-image-preview');
            const deleteBtn = document.getElementById('delete-category-image');
            const imageInput = document.getElementById('category-image');
            
            if (preview) {
                preview.innerHTML = '';
                if (category.image_url) {
                    let imageUrl = category.image_url;
                    if (imageUrl.startsWith('uploads/')) {
                        imageUrl = '../api/' + imageUrl;
                    }
                    const img = document.createElement('img');
                    img.src = imageUrl;
                    preview.appendChild(img);
                    
                    if (deleteBtn) {
                        deleteBtn.style.display = 'inline-block';
                    }
                    
                    // ?????? **???????: ?? ??? data-remove-image ?????? ??????????**
                    if (!imageUrl.includes('no-image')) {
                        imageInput.removeAttribute('data-remove-image');
                    }
                } else {
                    if (deleteBtn) {
                        deleteBtn.style.display = 'none';
                    }
                }
            }
        } else {
            title.textContent = 'Add New Category';
            document.getElementById('category-id').value = '';
            document.getElementById('category-description').value = '';
            document.getElementById('category-order').value = 0;
            
            // ?? ???????: ????? ????????? ??? true ??? ???????
            document.getElementById('category-active').checked = true;
            
            document.getElementById('category-image-preview').innerHTML = '';
            document.getElementById('delete-category-image').style.display = 'none';
        }
        
        modal.style.display = 'block';
    }

    openBookModal(book = null) {
    const modal = document.getElementById('book-modal');
    const title = document.getElementById('book-modal-title');
    const form = document.getElementById('book-form');
    
    if (!modal || !title || !form) return;
    
    form.reset();
    document.getElementById('book-image').removeAttribute('data-remove-image');
    
    document.getElementById('book-pdf')?.removeAttribute('required');
    document.getElementById('external-pdf-url')?.removeAttribute('required');
    document.getElementById('external-pdf-url-plus')?.removeAttribute('required');
    
    this.populateCategoryDropdowns();
    
    if (book) {
        title.textContent = 'Edit Book';
        document.getElementById('book-id').value = book.id;
        document.getElementById('book-title').value = book.title;
        document.getElementById('book-author').value = book.author || '';
        document.getElementById('book-language').value = book.language || '';
        document.getElementById('book-format').value = book.format || ''; // ?? NEW: Format field
        
        // ?? FIX CRITICAL: Set require login for download CORRECTLY
        const requireLoginChecked = book.require_login_for_download === true || book.require_login_for_download === 1 || book.require_login_for_download === '1';
        document.getElementById('book-require-login-for-download').checked = requireLoginChecked;
        
        // ?? FIX CRITICAL: Set active status CORRECTLY
        const isActiveChecked = book.is_active === true || book.is_active === 1 || book.is_active === '1';
        document.getElementById('book-active').checked = isActiveChecked;
        
        console.log('?? DEBUG - Loading book data:');
        console.log('   is_active from DB:', book.is_active, '-> Checked:', isActiveChecked);
        console.log('   require_login_for_download from DB:', book.require_login_for_download, '-> Checked:', requireLoginChecked);
        
        // ?? NEW: Set external download link values
        document.getElementById('book-external-download-link').value = book.external_download_link || '';
        
        // ?? NEW: Set external download protection values
        document.getElementById('book-external-download-protection-key').value = book.external_download_protection_key || '';
        
        // ?? ???????: ????? ???? ??????? ???? ????
        const isExternalDownloadProtected = book.is_external_download_protected === true || book.is_external_download_protected === 1 || book.is_external_download_protected === '1';
        document.getElementById('book-is-external-download-protected').checked = isExternalDownloadProtected;
        
        // ?????? ??????? ??????: ??????? ????? ???????? ?? ????? ????????
        document.getElementById('book-show-external-download-button').checked = book.show_external_download_button === true || book.show_external_download_button === 1 || book.show_external_download_button === '1';
        
        setTimeout(() => {
            document.getElementById('book-category').value = book.category_id || '';
            document.getElementById('book-profile-category').value = book.profile_category_id || '';
        }, 100);
        
        document.getElementById('book-summary').value = book.summary;
        document.getElementById('book-table-of-contents').value = book.table_of_contents || '';
        
        document.getElementById('book-page-count').value = book.page_count || '';
        
        // ?????? ??????? ??????? ????? ???oggles - ??????? ?????? ??????
        document.getElementById('book-show-file-size').checked = book.show_file_size === true || book.show_file_size === 1 || book.show_file_size === '1';
        document.getElementById('book-custom-file-size').value = book.custom_file_size || '';
        
        document.getElementById('book-show-read-button').checked = book.show_read_button === true || book.show_read_button === 1 || book.show_read_button === '1';
        document.getElementById('book-read-button-text').value = book.read_button_text || 'Read (Lire) Document';
        
        document.getElementById('book-show-download-extrait-button').checked = book.show_download_extrait_button === true || book.show_download_extrait_button === 1 || book.show_download_extrait_button === '1';
        document.getElementById('book-download-extrait-button-text').value = book.download_extrait_button_text || 'Download extrait';
        
        document.getElementById('book-taille-before-extrait').value = book.taille_before_extrait || '';
        document.getElementById('book-show-taille-before-extrait').checked = book.show_taille_before_extrait === true || book.show_taille_before_extrait === 1 || book.show_taille_before_extrait === '1';
        
        // ?????? ??????? ?????? ?? Download Protection - ?????? ?? ???? ??????? ???????
        const isProtected = book.is_protected === true || book.is_protected === 1 || book.is_protected === '1';
        document.getElementById('book-protection-enabled').checked = isProtected;
        document.getElementById('book-protection-key').value = book.protection_key || '';
        
        // ?? ????? ???oggles ??? ????????? - ??????? ?????
        this.toggleTailleExtraitGroup(book.show_taille_before_extrait === true || book.show_taille_before_extrait === 1 || book.show_taille_before_extrait === '1');
        this.toggleExternalDownloadLinkGroup(book.show_external_download_button === true || book.show_external_download_button === 1 || book.show_external_download_button === '1');
        this.toggleExternalDownloadProtectionGroup(isExternalDownloadProtected); // ?? ??????? ??????? ??????
        
        // ?? ?????? ?? ?????/????? ?????? ????? ???????
        const keyGroup = document.getElementById('protection-key-group');
        if (keyGroup) {
            keyGroup.style.display = isProtected ? 'block' : 'none';
        }
        
        document.getElementById('book-selected-pages-summary').value = book.selected_pages_summary || '';
        document.getElementById('book-selected-pages-toc').value = book.selected_pages_toc || '';
        document.getElementById('book-selected-pages-original').value = book.selected_pages_original || '';
        
        document.getElementById('book-institution').value = book.institution || '';
        document.getElementById('book-publication-year').value = book.publication_year || '';
        document.getElementById('book-keywords').value = book.keywords || '';
        
        // ?? DEBUG: ????? ??? ???oggles ??????
        console.log('?? LOADING BOOK DATA - TOGGLES STATUS:');
        console.log('   Show File Size:', book.show_file_size, '->', document.getElementById('book-show-file-size').checked);
        console.log('   Show Read Button:', book.show_read_button, '->', document.getElementById('book-show-read-button').checked);
        console.log('   Show Taille Before Extrait:', book.show_taille_before_extrait, '->', document.getElementById('book-show-taille-before-extrait').checked);
        console.log('   Show Download Extrait Button:', book.show_download_extrait_button, '->', document.getElementById('book-show-download-extrait-button').checked);
        console.log('   Enable Download Protection:', book.is_protected, '->', document.getElementById('book-protection-enabled').checked);
        console.log('   Show External Download Button:', book.show_external_download_button, '->', document.getElementById('book-show-external-download-button').checked);
        console.log('   Enable External Download Protection:', isExternalDownloadProtected, '->', document.getElementById('book-is-external-download-protected').checked);
        console.log('   Format:', book.format, '->', document.getElementById('book-format').value);
        console.log('   Is Active:', book.is_active, '->', document.getElementById('book-active').checked);
        console.log('   Require Login for Download:', book.require_login_for_download, '->', document.getElementById('book-require-login-for-download').checked);

        if (book.is_external) {
    if (book.is_external_plus) {
        document.querySelector('input[name="pdf_url_option"][value="url_plus"]').checked = false;
        document.getElementById('file-upload-group').style.display = 'none';
        document.getElementById('external-url-group').style.display = 'none';
        document.getElementById('external-url-plus-group').style.display = 'block';
        document.getElementById('external-pdf-url-plus').value = book.pdf_url;
    } else if (book.is_external_plus && !book.pdf_url) {
        document.getElementById('external-pdf-url-plus').value = 'https://www.acadlib.com/app-panel/document.pdf';
    } else {
                document.querySelector('input[name="pdf_url_option"][value="url"]').checked = true;
                document.getElementById('file-upload-group').style.display = 'none';
                document.getElementById('external-url-group').style.display = 'block';
                document.getElementById('external-url-plus-group').style.display = 'none';
                document.getElementById('external-pdf-url').value = book.pdf_url;
            }
        } else {
            document.querySelector('input[name="pdf_url_option"][value="file"]').checked = true;
            document.getElementById('file-upload-group').style.display = 'block';
            document.getElementById('external-url-group').style.display = 'none';
            document.getElementById('external-url-plus-group').style.display = 'none';
            
            document.getElementById('external-pdf-url').value = '';
            document.getElementById('external-pdf-url-plus').value = '';
        }
        
        const preview = document.getElementById('book-image-preview');
        const deleteBtn = document.getElementById('delete-book-image');
        const imageInput = document.getElementById('book-image');
        
        if (preview) {
            preview.innerHTML = '';
            if (book.image_url) {
                let imageUrl = book.image_url;
                if (imageUrl.startsWith('uploads/')) {
                    imageUrl = '../api/' + imageUrl;
                }
                const img = document.createElement('img');
                img.src = imageUrl;
                preview.appendChild(img);
                
                if (deleteBtn) {
                    deleteBtn.style.display = 'inline-block';
                }
                
                // ?????? **???????: ?? ??? data-remove-image ?????? ??????????**
                if (!imageUrl.includes('no-image.jpg')) {
                    imageInput.removeAttribute('data-remove-image');
                }
            } else {
                if (deleteBtn) {
                    deleteBtn.style.display = 'none';
                }
            }
        }
    } else {
    title.textContent = 'Add New Book';
    document.getElementById('book-id').value = '';
    document.getElementById('book-image-preview').innerHTML = '';
    document.getElementById('book-language').value = '';
    document.getElementById('book-format').value = ''; // ?? NEW: Reset format field
    document.getElementById('external-pdf-url').value = '';
    document.getElementById('external-pdf-url-plus').value = 'https://www.acadlib.com/app-panel/document.pdf';
    document.getElementById('book-table-of-contents').value = '';
        
        // ?? NEW: Reset external download link fields
        document.getElementById('book-external-download-link').value = '';
        document.getElementById('book-show-external-download-button').checked = false;
        document.getElementById('book-external-download-protection-key').value = '';
        document.getElementById('book-is-external-download-protected').checked = false;
        
        document.getElementById('book-page-count').value = '';
        
        document.getElementById('book-selected-pages-summary').value = '';
        document.getElementById('book-selected-pages-toc').value = '';
        document.getElementById('book-selected-pages-original').value = '';
        
        // ?? FIX CRITICAL: Set default values for new books
        document.getElementById('book-active').checked = true; // Default: Active
        document.getElementById('book-require-login-for-download').checked = false; // Default: Require Login
        
        console.log('?? DEBUG - Setting default values for new book:');
        console.log('   is_active: true (default)');
        console.log('   require_login_for_download: true (default)');
        
        // ?? ??????? ???????: ????? ???? ???oggles ??? ????? ???????
        document.getElementById('book-show-file-size').checked = true;
        document.getElementById('book-custom-file-size').value = '';
        
        document.getElementById('book-show-read-button').checked = true;
        document.getElementById('book-read-button-text').value = 'Read (Lire) Document';
        
        document.getElementById('book-show-download-extrait-button').checked = true;
        document.getElementById('book-download-extrait-button-text').value = 'Download extrait';
        
        document.getElementById('book-taille-before-extrait').value = '';
        document.getElementById('book-show-taille-before-extrait').checked = false;
        
        // ?? ???????: ????? ?????? ?????????? ?????????
        this.toggleTailleExtraitGroup(false);
        this.toggleExternalDownloadLinkGroup(false);
        this.toggleExternalDownloadProtectionGroup(false);
        
        // ?????? ??????? ??????: ????? ??????? ??? FALSE ?????????
        document.getElementById('book-protection-enabled').checked = false;
        document.getElementById('book-protection-key').value = '';
        
        // ?? ????? ????? ?????? ???????
        const keyGroup = document.getElementById('protection-key-group');
        if (keyGroup) {
            keyGroup.style.display = 'none';
        }
        
        document.querySelector('input[name="pdf_url_option"][value="file"]').checked = true;
        document.getElementById('file-upload-group').style.display = 'block';
        document.getElementById('external-url-group').style.display = 'none';
        document.getElementById('external-url-plus-group').style.display = 'none';
        
        document.getElementById('delete-book-image').style.display = 'none';
        
        this.populateCategoryDropdowns();
    }
    
    this.setupPdfSourceToggle();
    
    modal.style.display = 'block';
}

    openProfileCategoryModal(category = null) {
        const modal = document.getElementById('profile-category-modal');
        const title = document.getElementById('profile-category-modal-title');
        const form = document.getElementById('profile-category-form');
        
        if (!modal || !title || !form) return;
        
        form.reset();
        document.getElementById('profile-category-image').removeAttribute('data-remove-image');
        
        if (category) {
            title.textContent = 'Edit Profile Category';
            document.getElementById('profile-category-id').value = category.id;
            document.getElementById('profile-category-name').value = category.name;
            document.getElementById('profile-category-description').value = category.description || '';
            document.getElementById('profile-category-order').value = category.display_order || 0;
            
            // ?????? ??????? ???????: ????? ???? ??? checkbox ????? ??? ?????? ???????? ?? ????? ????????
            const isActiveCheckbox = document.getElementById('profile-category-active');
            if (isActiveCheckbox) {
                // ????? ?????? ??? boolean ???? ????
                const isActive = Boolean(category.is_active && category.is_active !== 0 && category.is_active !== '0');
                isActiveCheckbox.checked = isActive;
                console.log('?? Setting profile category active status:', isActive, 'from database value:', category.is_active, 'type:', typeof category.is_active);
            }
            
            const preview = document.getElementById('profile-category-image-preview');
            const deleteBtn = document.getElementById('delete-profile-category-image');
            const imageInput = document.getElementById('profile-category-image');
            
            if (preview) {
                preview.innerHTML = '';
                if (category.image_url) {
                    let imageUrl = category.image_url;
                    if (imageUrl.startsWith('uploads/')) {
                        imageUrl = '../api/' + imageUrl;
                    }
                    const img = document.createElement('img');
                    img.src = imageUrl;
                    preview.appendChild(img);
                    
                    if (deleteBtn) {
                        deleteBtn.style.display = 'inline-block';
                    }
                    
                    // ?????? **???????: ?? ??? data-remove-image ?????? ??????????**
                    if (!imageUrl.includes('no-image')) {
                        imageInput.removeAttribute('data-remove-image');
                    }
                } else {
                    if (deleteBtn) {
                        deleteBtn.style.display = 'none';
                    }
                }
            }
        } else {
            title.textContent = 'Add Profile Category';
            document.getElementById('profile-category-id').value = '';
            document.getElementById('profile-category-description').value = '';
            document.getElementById('profile-category-order').value = 0;
            
            // ?? ???????: ????? ????????? ??? true ??? ???????
            document.getElementById('profile-category-active').checked = true;
            
            document.getElementById('profile-category-image-preview').innerHTML = '';
            document.getElementById('delete-profile-category-image').style.display = 'none';
        }
        
        modal.style.display = 'block';
    }

    closeModals() {
        ['category-modal', 'book-modal', 'profile-category-modal'].forEach(modalId => {
            const modal = document.getElementById(modalId);
            if (modal) {
                modal.style.display = 'none';
            }
        });
    }

    formatFileSize(bytes) {
        if (!bytes || bytes === 0) return '0 B';
        
        const sizes = ['B', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(1024));
        return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
    }

    // ?? UPDATE: ????? ???? saveCategory ?????? ??????????
    async saveCategory() {
        const formData = new FormData();
        const id = document.getElementById('category-id').value;
        const name = document.getElementById('category-name').value;
        const description = document.getElementById('category-description').value;
        const displayOrder = document.getElementById('category-order').value;
        
        // ?????? ??????? ???????: ?????? ??? ???? ??? checkbox ???? ????
        const isActive = document.getElementById('category-active').checked;
        
        const imageFile = document.getElementById('category-image').files[0];
        const removeImage = document.getElementById('category-image').getAttribute('data-remove-image') === 'true';

        console.log('?? Saving category - Active status:', isActive, 'ID:', id);

        if (!name) {
            alert('Please enter category name');
            return;
        }

        formData.append('name', name);
        formData.append('description', description);
        formData.append('display_order', displayOrder);
        
        // ?????? ??????? ???????: ????? ?????? ??? 'true' ?? 'false'
        formData.append('is_active', isActive.toString());
        
        if (imageFile) {
            formData.append('image', imageFile);
        }
        
        // ?????? **??????? ??????: ????? remove_image ??? ??? ???? ???? ???? ????? ????? ????????**
        if (removeImage) {
            formData.append('remove_image', 'true');
            console.log('??? Sending remove_image flag for custom category image');
        }

        if (id) {
            formData.append('id', id);
        }

        try {
            const url = `${API_BASE}/categories.php`;
            console.log('?? Sending category data with active:', isActive);
            
            const response = await fetch(url, {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Server error');
            }

            console.log('? Category saved successfully with active status:', isActive);
            this.closeModals();
            await this.loadCategories(); // ?? ?????? ??? ????? ?????????? ????????
            alert('Category saved successfully');
            
        } catch (error) {
            console.error('Error saving category:', error);
            alert('Error while saving: ' + error.message);
        }
    }

   // ?? UPDATE: ????? ???? saveBook ?????? ??????????
async saveBook(e) {
    console.log('?? saveBook function STARTED');
    
    if (e) {
        e.preventDefault();
        console.log('? Event prevented');
    }

    document.getElementById('book-pdf')?.removeAttribute('required');
    document.getElementById('external-pdf-url')?.removeAttribute('required');
    document.getElementById('external-pdf-url-plus')?.removeAttribute('required');

    const id = document.getElementById('book-id').value;
    const title = document.getElementById('book-title')?.value?.trim() || '';
    const author = document.getElementById('book-author')?.value?.trim() || '';
    const language = document.getElementById('book-language')?.value?.trim() || '';
    const format = document.getElementById('book-format')?.value?.trim() || '';
    const categoryId = document.getElementById('book-category')?.value || '';
    const profileCategoryId = document.getElementById('book-profile-category')?.value || '';
    const summary = document.getElementById('book-summary')?.value?.trim() || '';
    const pdf_url_option = document.querySelector('input[name="pdf_url_option"]:checked')?.value || 'file';
    const external_pdf_url = document.getElementById('external-pdf-url')?.value?.trim() || '';
    const external_pdf_url_plus = document.getElementById('external-pdf-url-plus')?.value?.trim() || '';
    const imageFile = document.getElementById('book-image')?.files[0];
    const pdfFile = document.getElementById('book-pdf')?.files[0];
    const table_of_contents = document.getElementById('book-table-of-contents')?.value?.trim() || '';
    const removeImage = document.getElementById('book-image')?.getAttribute('data-remove-image') === 'true';
    
    const stockage = document.getElementById('book-stockage')?.value?.trim() || '';
    
    // ?? NEW: External download link fields
    const external_download_link = document.getElementById('book-external-download-link')?.value?.trim() || '';
    const show_external_download_button = document.getElementById('book-show-external-download-button')?.checked === true;
    
    // ?? NEW: External download protection fields
    const external_download_protection_key = document.getElementById('book-external-download-protection-key')?.value?.trim() || '';
    const is_external_download_protected = document.getElementById('book-is-external-download-protected')?.checked === true;
    
    // ?? FIX CRITICAL: ?????? ??? ??? ?????? ???? ????
    const is_active = document.getElementById('book-active')?.checked === true;
    const require_login_for_download = document.getElementById('book-require-login-for-download')?.checked === true;
    
    const page_count_input = document.getElementById('book-page-count')?.value?.trim() || '';
    const page_count = page_count_input ? parseInt(page_count_input) : null;
    
    let useDefaultImage = false;
    
    if (!id && !imageFile && !removeImage) {
        useDefaultImage = true;
        console.log('??? No image provided - using default image automatically');
    }
    
    // ?????? ??????? ??????? ????? ???oggles:
    const showFileSize = document.getElementById('book-show-file-size')?.checked === true;
    const showReadButton = document.getElementById('book-show-read-button')?.checked === true;
    const showTailleBeforeExtrait = document.getElementById('book-show-taille-before-extrait')?.checked === true;
    const showDownloadExtraitButton = document.getElementById('book-show-download-extrait-button')?.checked === true;
    
    // ?????? ??????? ??????: ?????? ??? ??????? ???? ????
    const isProtected = document.getElementById('book-protection-enabled')?.checked === true;
    let protectionKey = document.getElementById('book-protection-key')?.value?.trim() || '';

    // ??? ???? ??????? ??? ?????? ???? ?? ?? ????? ??????? ????
    if (!isProtected) {
        protectionKey = '';
    }

    const customFileSize = document.getElementById('book-custom-file-size')?.value?.trim() || '';
    const readButtonText = document.getElementById('book-read-button-text')?.value?.trim() || 'Read (Lire) Document';
    
    const tailleBeforeExtrait = document.getElementById('book-taille-before-extrait')?.value?.trim() || '';
    
    // 🔥 IMPORTANT: Page Extraction Settings - ??? ??????? ???????
    const selectedPagesSummary = document.getElementById('book-selected-pages-summary')?.value?.trim() || '';
    const selectedPagesTOC = document.getElementById('book-selected-pages-toc')?.value?.trim() || '';
    const selectedPagesOriginal = document.getElementById('book-selected-pages-original')?.value?.trim() || '';
    
    const institution = document.getElementById('book-institution')?.value?.trim() || '';
    const publicationYear = document.getElementById('book-publication-year')?.value?.trim() || '';
    const keywords = document.getElementById('book-keywords')?.value?.trim() || '';
    
    const downloadExtraitButtonText = document.getElementById('book-download-extrait-button-text')?.value?.trim() || 'Download extrait';

    console.log('?? Save Book Debug - TOGGLES STATUS:');
    console.log('   Show File Size:', showFileSize);
    console.log('   Show Read Button:', showReadButton);
    console.log('   Show Taille Before Extrait:', showTailleBeforeExtrait);
    console.log('   Show Download Extrait Button:', showDownloadExtraitButton);
    console.log('   Enable Download Protection:', isProtected);
    console.log('   Show External Download Button:', show_external_download_button);
    console.log('   Enable External Download Protection:', is_external_download_protected);
    console.log('   External Download Protection Key:', external_download_protection_key);
    console.log('   Protection Key:', protectionKey);
    console.log('   Format:', format);
    
    // ?? FIX: Debug log for critical checkboxes
    console.log('?? DEBUG - CRITICAL CHECKBOX VALUES:');
    console.log('   Is Active (book-active):', is_active);
    console.log('   Require Login for Download (book-require-login-for-download):', require_login_for_download);
    console.log('   Checkbox elements found:');
    console.log('     - book-active:', document.getElementById('book-active'));
    console.log('     - book-require-login-for-download:', document.getElementById('book-require-login-for-download'));

    // ?? NEW: FINAL CHECK before sending
    console.log('?? FINAL CHECK before sending:');
    console.log('   is_active:', document.getElementById('book-active')?.checked);
    console.log('   require_login_for_download:', document.getElementById('book-require-login-for-download')?.checked);

    // ???? ????? ?????? ?? ?????
    const finalIsActive = document.getElementById('book-active')?.checked;
    const finalRequireLogin = document.getElementById('book-require-login-for-download')?.checked;

    if (finalIsActive === undefined || finalRequireLogin === undefined) {
        console.error('? ERROR: Checkbox values are undefined!');
        alert('Error: Checkbox values are missing. Please refresh the page and try again.');
        return;
    }

    // 🔥 IMPORTANT: Page Extraction Settings Log
    console.log('?? PAGE EXTRACTION SETTINGS:');
    console.log('   Storage Pages (Original PDF):', selectedPagesOriginal);
    console.log('   TOC Pages:', selectedPagesTOC);
    console.log('   Summary Pages:', selectedPagesSummary);

    if (!title) {
        alert('Please enter book title');
        document.getElementById('book-title')?.focus();
        return;
    }

    if (!categoryId && !profileCategoryId) {
        alert('Please select at least one category (regular or profile)');
        return;
    }

    if (!id) {
        if (pdf_url_option === 'file') {
            if (!pdfFile) {
                alert('Please select a PDF file for the new book');
                return;
            }
        } else if (pdf_url_option === 'url') {
            if (!external_pdf_url) {
                alert('Please enter external PDF URL');
                document.getElementById('external-pdf-url')?.focus();
                return;
            }
        } else if (pdf_url_option === 'url_plus') {
            if (!external_pdf_url_plus) {
                alert('Please enter external PDF URL for External Link Plus');
                document.getElementById('external-pdf-url-plus')?.focus();
                return;
            }
        }
    } else {
        console.log('?? Updating existing book - no PDF validation required');
    }

    console.log('? All validations passed');

    const formData = new FormData();
    
    formData.append('title', title);
    if (author) formData.append('author', author);
    formData.append('language', language);
    formData.append('format', format);
    formData.append('summary', summary);
    formData.append('pdf_url_option', pdf_url_option);
    formData.append('table_of_contents', table_of_contents);
    
    // ?? NEW: Add external download link fields
    formData.append('external_download_link', external_download_link);
    formData.append('show_external_download_button', show_external_download_button.toString());
    
    // ?? NEW: Add external download protection fields
    formData.append('external_download_protection_key', external_download_protection_key);
    formData.append('is_external_download_protected', is_external_download_protected.toString());
    
    // ?? FIX CRITICAL: Add active status and require login fields CORRECTLY
    formData.append('is_active', is_active.toString());
    formData.append('require_login_for_download', require_login_for_download.toString());
    
    if (page_count !== null) {
        formData.append('page_count', page_count.toString());
    }
    
    if (useDefaultImage) {
        formData.append('use_default_image', 'true');
    }
    
    // ?? ????? ???? ???oggles ????????:
    formData.append('show_file_size', showFileSize.toString());
    formData.append('custom_file_size', customFileSize);
    
    formData.append('show_read_button', showReadButton.toString());
    formData.append('read_button_text', readButtonText);
    
    formData.append('stockage', stockage);
    
    formData.append('show_download_extrait_button', showDownloadExtraitButton.toString());
    formData.append('download_extrait_button_text', downloadExtraitButtonText);
    
    formData.append('taille_before_extrait', tailleBeforeExtrait);
    formData.append('show_taille_before_extrait', showTailleBeforeExtrait.toString());
    
    // 🔥 IMPORTANT: ????? ???? ????? ???????
    formData.append('selected_pages_summary', selectedPagesSummary);
    formData.append('selected_pages_toc', selectedPagesTOC);
    formData.append('selected_pages_original', selectedPagesOriginal);
    
    formData.append('institution', institution);
    formData.append('publication_year', publicationYear);
    formData.append('keywords', keywords);

    // ?? ??????? ??????? ?? Download Protection:
    formData.append('is_protected', isProtected.toString());
    formData.append('protection_key', protectionKey);

    if (categoryId) formData.append('category_id', categoryId);
    if (profileCategoryId) formData.append('profile_category_id', profileCategoryId);
    
    if (pdf_url_option === 'url') {
        formData.append('external_pdf_url', external_pdf_url);
    } else if (pdf_url_option === 'url_plus') {
        formData.append('external_pdf_url_plus', external_pdf_url_plus);
    } else if (pdfFile) {
        formData.append('pdf', pdfFile);
    }
    
    if (imageFile) {
        formData.append('image', imageFile);
    }
    
    // ?????? **??????? ??????: ????? remove_image ??? ??? ???? ???? ???? ????? ????? ????????**
    if (removeImage) {
        formData.append('remove_image', 'true');
        console.log('??? Sending remove_image flag for custom book image');
    }

    if (id) {
        formData.append('id', id);
    }

    console.log('?? FormData prepared, sending request...');
    console.log('?? Protection Status - isProtected:', isProtected, 'protectionKey:', protectionKey);
    console.log('?? External Download Protection Status - is_external_download_protected:', is_external_download_protected, 'external_download_protection_key:', external_download_protection_key);
    console.log('?? Format Value:', format);
    
    // ?? FIX: Debug log for critical values being sent
    console.log('?? CRITICAL VALUES BEING SENT:');
    console.log('   is_active:', is_active);
    console.log('   require_login_for_download:', require_login_for_download);
    
    // 🔥 IMPORTANT: Log page extraction settings being sent
    console.log('?? PAGE EXTRACTION SETTINGS BEING SENT:');
    console.log('   Storage Pages:', selectedPagesOriginal);
    console.log('   TOC Pages:', selectedPagesTOC);
    console.log('   Summary Pages:', selectedPagesSummary);

    try {
        const url = `${API_BASE}/books.php`;
        console.log('?? Sending to:', url);

        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });

        console.log('?? Response status:', response.status);

        const result = await response.json();
        console.log('?? Server response:', result);

        if (!response.ok) {
            // ?? NEW: Check for duplicate title error and show message to user
            if (result.error && (result.error.includes('already exists') || result.error.includes('same title'))) {
                alert(' ' + result.error);
                console.log('? Duplicate title detected - keeping modal open for correction');
                return; // Stop execution, don't close modal
            }
            throw new Error(result.error || `Server error: ${response.status}`);
        }

        console.log('? Book saved successfully, closing modal...');
        console.log('?? Final values saved:');
        console.log('   is_active:', is_active);
        console.log('   require_login_for_download:', require_login_for_download);
        
        this.closeModals();
        await this.loadBooks();
        
        alert('Book saved successfully!');
        
    } catch (error) {
        console.error('? Error saving book:', error);
        alert('Error while saving: ' + error.message);
    }
}

    // ?? UPDATE: ????? ???? saveProfileCategory ?????? ??????????
    async saveProfileCategory() {
        const formData = new FormData();
        const id = document.getElementById('profile-category-id').value;
        const name = document.getElementById('profile-category-name').value;
        const description = document.getElementById('profile-category-description').value;
        const displayOrder = document.getElementById('profile-category-order').value;
        
        // ?????? ??????? ???????: ?????? ??? ???? ??? checkbox ???? ????
        const isActive = document.getElementById('profile-category-active').checked;
        
        const imageFile = document.getElementById('profile-category-image').files[0];
        const removeImage = document.getElementById('profile-category-image').getAttribute('data-remove-image') === 'true';

        console.log('?? Saving profile category - Active status:', isActive, 'ID:', id);

        if (!name) {
            alert('Please enter category name');
            return;
        }

        formData.append('name', name);
        formData.append('description', description);
        formData.append('display_order', displayOrder);
        
        // ?????? ??????? ???????: ????? ?????? ??? 'true' ?? 'false'
        formData.append('is_active', isActive.toString());
        
        if (imageFile) {
            formData.append('image', imageFile);
        }
        
        // ?????? **??????? ??????: ????? remove_image ??? ??? ???? ???? ???? ????? ????? ????????**
        if (removeImage) {
            formData.append('remove_image', 'true');
            console.log('??? Sending remove_image flag for custom profile category image');
        }

        if (id) {
            formData.append('id', id);
        }

        try {
            const url = `${API_BASE}/profile_categories.php`;
            console.log('?? Sending profile category data with active:', isActive);
            
            const response = await fetch(url, {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Server error');
            }

            console.log('? Profile category saved successfully with active status:', isActive);
            this.closeModals();
            await this.loadProfileCategories(); // ?? ?????? ??? ????? ?????????? ????????
            this.renderProfileCategories();
            this.populateCategoryDropdowns();
            alert('Profile category saved successfully');
            
        } catch (error) {
            console.error('Error saving profile category:', error);
            alert('Error while saving: ' + error.message);
        }
    }

    // ?? UPDATE: ????? ???? deleteCategory ?????? ??????????
    async deleteCategory(id) {
        if (!confirm('Are you sure you want to delete this category?')) return;

        try {
            const response = await fetch(`${API_BASE}/categories.php`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `id=${id}`
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Server error');
            }

            await this.loadCategories(); // ?? ?????? ??? ????? ?????????? ????????
            alert('Deleted successfully');
            
        } catch (error) {
            console.error('Error deleting category:', error);
            alert('Error while deleting: ' + error.message);
        }
    }

    // ?? UPDATE: ????? ???? deleteBook ?????? ??????????
    async deleteBook(id) {
        if (!confirm('Are you sure you want to delete this book?')) return;

        try {
            const response = await fetch(`${API_BASE}/books.php`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `id=${id}`
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Server error');
            }

            await this.loadBooks(); // ?? ?????? ??? ????? ?????????? ????????
            alert('Deleted successfully');
            
        } catch (error) {
            console.error('Error deleting book:', error);
            alert('Error while deleting: ' + error.message);
        }
    }

    // ?? UPDATE: ????? ???? deleteProfileCategory ?????? ??????????
    async deleteProfileCategory(id) {
        if (!confirm('Are you sure you want to delete this category?')) return;

        try {
            const response = await fetch(`${API_BASE}/profile_categories.php`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `id=${id}`
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Server error');
            }

            await this.loadProfileCategories(); // ?? ?????? ??? ????? ?????????? ????????
            this.renderProfileCategories();
            this.populateCategoryDropdowns();
            alert('Profile category deleted successfully');
            
        } catch (error) {
            console.error('Error deleting profile category:', error);
            alert('Error while deleting: ' + error.message);
        }
    }

    editCategory(id) {
        const category = this.categories.find(c => c.id == id);
        if (category) {
            console.log('?? Editing category ID:', category.id, 'Active status:', category.is_active);
            this.openCategoryModal(category);
        }
    }

    editBook(id) {
        const book = this.books.find(b => b.id == id);
        if (book) {
            console.log('?? Editing book ID:', book.id);
            this.openBookModal(book);
        }
    }

    editProfileCategory(id) {
        const category = this.profileCategories.find(c => c.id == id);
        if (category) {
            console.log('?? Editing profile category ID:', category.id, 'Active status:', category.is_active);
            this.openProfileCategoryModal(category);
        }
    }

    // ???? ?????? ?? ?????/????? ?????? Taille Extrait
    toggleTailleExtraitGroup(isVisible) {
        const tailleGroup = document.getElementById('taille-before-extrait-group');
        if (tailleGroup) {
            tailleGroup.style.display = isVisible ? 'block' : 'none';
        }
        
        if (isVisible) {
            console.log('? Taille Extrait activated');
        } else {
            console.log('? Taille Extrait deactivated');
        }
    }
    
    // ?? NEW: ???? ?????? ?? ?????/????? ?????? ?????? ???????
    toggleExternalDownloadLinkGroup(isVisible) {
        const linkGroup = document.getElementById('external-download-link-group');
        if (linkGroup) {
            linkGroup.style.display = isVisible ? 'block' : 'none';
        }
        
        if (isVisible) {
            console.log('? External Download Link activated');
        } else {
            console.log('? External Download Link deactivated');
        }
    }
    
    // ?? NEW: ???? ?????? ?? ?????/????? ?????? ????? ?????? ???????
    toggleExternalDownloadProtectionGroup(isVisible) {
        const protectionGroup = document.getElementById('external-download-protection-group');
        if (protectionGroup) {
            protectionGroup.style.display = isVisible ? 'block' : 'none';
            
            // ??? ?? ????? ???????? ???? ??? ???????
            if (!isVisible) {
                document.getElementById('book-external-download-protection-key').value = '';
            }
            
            if (isVisible) {
                console.log('? External Download Protection activated');
            } else {
                console.log('? External Download Protection deactivated');
            }
        } else {
            console.error('? external-download-protection-group not found');
        }
    }
}

// Initialize the app when the page loads
document.addEventListener('DOMContentLoaded', () => {
    console.log('Page loaded, initializing app...');
    console.log('?? PDF Processing with Ghostscript support enabled');
    console.log('?? Large file support (up to 100MB) activated');
    console.log('?? Pagination system activated - 30 items per page');
    console.log('?? Real-time analytics system activated for all sections');
    console.log('??? Default image protection system activated');
    window.adminApp = new AdminApp();
});