<?php
// =============================================
// ?? LITESPEED ULTIMATE FIX - CLEAN ALL OUTPUT
// =============================================

// ????? ???? ????? ??????? ??? ??????? ???
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// ????? ???? buffers
while (ob_get_level() > 0) {
    ob_end_clean();
}

// ????? buffer ???? ?? ???? ????
ob_start();

// ??? ?????? ?????
header('Content-Type: application/json; charset=utf-8');

// ????? ???? ????? ????? ????????
if (function_exists('apache_setenv')) {
    @apache_setenv('no-gzip', 1);
}
@ini_set('zlib.output_compression', 0);
@ini_set('output_buffering', 0);
@ini_set('implicit_flush', 0);

// =============================================
// ?? ????? ?????? ??????
// =============================================
include 'config.php';

// ???? ?????? ??????? ?????
function clean_json_output($data) {
    // ????? ?????
    while (ob_get_level() > 0) {
        ob_end_clean();
    }
    
    // ????? JSON ????
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function json_success($message, $data = []) {
    $response = array_merge(['success' => true, 'message' => $message], $data);
    clean_json_output($response);
}

function json_error($message) {
    http_response_code(400); // ?? ????? ??? ??? ?????
    clean_json_output(['success' => false, 'error' => $message]);
}

ob_start();

// ????? ???????
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/debug.log');

// ?? NEW: ???? ???????? ????????
function getUploadDirectories() {
    return [
        'original' => 'uploads/pdfs/original/',
        'display' => 'uploads/pdfs/display/', 
        'extracted_toc' => 'uploads/pdfs/extracted/toc/',
        'extracted_summary' => 'uploads/pdfs/extracted/summary/',
        'images' => 'uploads/images/'
    ];
}

// ?? NEW: ???? ?????? ???????? ??? ?? ??? ??????
function createUploadDirectories() {
    $dirs = getUploadDirectories();
    
    foreach ($dirs as $type => $path) {
        if (!is_dir($path)) {
            mkdir($path, 0755, true);
            error_log("? Created directory: $path");
        }
    }
}

// ?? NEW: ???? ?????? ??? ???? ????? ????? ??? ?????
function getTargetPath($fileName, $fileType) {
    $dirs = getUploadDirectories();
    
    if (!isset($dirs[$fileType])) {
        error_log("? Unknown file type: $fileType");
        return false;
    }
    
    $dir = $dirs[$fileType];
    
    // ????? ?????? ??? ?? ??? ???????
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    
    return $dir . $fileName;
}

// ????? ?? ??? ?????
if ($_SERVER['REQUEST_METHOD'] == 'GET' && isset($_GET['id'])) {
    error_log("=== DOWNLOAD REQUEST ===");
    error_log("Time: " . date('Y-m-d H:i:s'));
    error_log("Book ID: " . $_GET['id']);
    error_log("User Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'));
    error_log("Remote IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'Unknown'));
}

error_log("=== DEBUG: books.php accessed ===");

// ????? ?????? PDF
$fpdfPath = __DIR__ . '/fpdf/fpdf.php';
$fpdiPath = __DIR__ . '/fpdi/src/autoload.php';

error_log("Checking FPDF path: " . $fpdfPath);
error_log("FPDF exists: " . (file_exists($fpdfPath) ? 'YES' : 'NO'));
error_log("Checking FPDI path: " . $fpdiPath);
error_log("FPDI exists: " . (file_exists($fpdiPath) ? 'YES' : 'NO'));

if (file_exists($fpdfPath) && file_exists($fpdiPath)) {
    require_once $fpdfPath;
    require_once $fpdiPath;
    error_log("? Libraries loaded successfully");
} else {
    error_log("? ERROR: Library files not found!");
}

use setasign\Fpdi\Fpdi;

// ?? NEW: ???? ?????? ?? ???? Ghostscript
function isGhostscriptAvailable() {
    try {
        $output = shell_exec("gs --version 2>&1");
        $available = !empty(trim($output));
        error_log("?? Ghostscript available: " . ($available ? 'YES - ' . trim($output) : 'NO'));
        return $available;
    } catch (Exception $e) {
        error_log("? Ghostscript check error: " . $e->getMessage());
        return false;
    }
}

// ?????? ?? Ghostscript ??? ????? ?????
error_log("=== GHOSTSCRIPT AVAILABILITY CHECK ===");
$ghostscriptAvailable = isGhostscriptAvailable();
error_log("Ghostscript will be used for large files: " . ($ghostscriptAvailable ? 'YES' : 'NO'));

// ?? NEW: ???? ?????? ??? ?????
function formatFileSize($bytes) {
    if ($bytes == 0) return '0 B';
    
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes, 1024));
    
    return round($bytes / pow(1024, $i), 2) . ' ' . $units[$i];
}

// ?? NEW: ???? ????? taille_before_extrait ????????
function calculateAutoTailleBeforeExtrait($filePath, $pageCount) {
    if (!file_exists($filePath)) {
        error_log("? File not found for auto taille calculation: $filePath");
        return '';
    }
    
    try {
        $fileSize = filesize($filePath);
        $sizeFormatted = formatFileSize($fileSize);
        
        // ????? ????: "X pages - Y MB"
        $autoTaille = $pageCount . ' pages - ' . $sizeFormatted;
        error_log("? Auto-calculated taille_before_extrait: $autoTaille");
        
        return $autoTaille;
    } catch (Exception $e) {
        error_log("? Error calculating auto taille: " . $e->getMessage());
        return '';
    }
}

// ???? ?????: ??? ??????? ??????? ????? ???? ??????? ?????
function deleteOldExtractedFiles($pdo, $bookId, $selected_pages_toc, $selected_pages_summary) {
    error_log("??? Checking for old extracted files to delete...");
    
    $stmt = $pdo->prepare('SELECT extracted_pdf_url, extracted_summary_pdf_url FROM books WHERE id = ?');
    $stmt->execute([$bookId]);
    $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existingBook) {
        // ??? ??? TOC ??? ?? ??? ???? ????? ?????
        if (empty($selected_pages_toc) && !empty($existingBook['extracted_pdf_url'])) {
            $oldTocPath = $existingBook['extracted_pdf_url'];
            if (file_exists($oldTocPath) && !filter_var($oldTocPath, FILTER_VALIDATE_URL)) {
                if (unlink($oldTocPath)) {
                    error_log("? Deleted old TOC PDF (no pages selected): $oldTocPath");
                } else {
                    error_log("? Failed to delete old TOC PDF: $oldTocPath");
                }
            }
        }
        
        // ??? ??? Summary ??? ?? ??? ???? ????? ?????
        if (empty($selected_pages_summary) && !empty($existingBook['extracted_summary_pdf_url'])) {
            $oldSummaryPath = $existingBook['extracted_summary_pdf_url'];
            if (file_exists($oldSummaryPath) && !filter_var($oldSummaryPath, FILTER_VALIDATE_URL)) {
                if (unlink($oldSummaryPath)) {
                    error_log("? Deleted old Summary PDF (no pages selected): $oldSummaryPath");
                } else {
                    error_log("? Failed to delete old Summary PDF: $oldSummaryPath");
                }
            }
        }
    }
}

// ?? NEW: ???? ????? ?????? ????? ???????
function downloadExternalPdf($externalUrl, $uploadDir) {
    error_log("???? ENHANCED Downloading external PDF: $externalUrl");
    
    // ????? URL
    $externalUrl = trim($externalUrl);
    
    if (!filter_var($externalUrl, FILTER_VALIDATE_URL)) {
        error_log("? Invalid URL format: $externalUrl");
        return false;
    }
    
    try {
        // ????? CURL ????
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $externalUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 45, // ????? ?????
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HEADER => false,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_BUFFERSIZE => 128000, // ????? buffer size
        ]);
        
        error_log("?? Sending HTTP request to: $externalUrl");
        $pdfData = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        $effectiveUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
        $downloadSize = curl_getinfo($ch, CURLINFO_SIZE_DOWNLOAD);
        
        curl_close($ch);
        
        error_log("?? Download Results - HTTP: $httpCode, Type: $contentType, Size: $downloadSize, Final URL: $effectiveUrl");

        // ?????? ?? ???? ???????
        if ($httpCode !== 200) {
            error_log("? Download failed. HTTP Code: $httpCode");
            return false;
        }
        
        if (empty($pdfData) || strlen($pdfData) < 100) { // ??? PDF ?? ???? ?? ???? ??? ?? 100 ????
            error_log("? Download failed. Empty or too small file: " . strlen($pdfData) . " bytes");
            return false;
        }
        
        // ?????? ?? ??? ??????? (?????? ????)
        $isPdfContent = (
            strpos($contentType, 'application/pdf') !== false ||
            strpos($contentType, 'application/octet-stream') !== false ||
            strpos($contentType, 'binary/octet-stream') !== false ||
            preg_match('/\.pdf$/i', $effectiveUrl) ||
            substr($pdfData, 0, 4) === '%PDF' // ?????? ?? ????? PDF
        );
        
        if (!$isPdfContent) {
            error_log("? Not a PDF file. Content-Type: $contentType, PDF Signature: " . (substr($pdfData, 0, 4) === '%PDF' ? 'YES' : 'NO'));
            return false;
        }
        
        // ????? ??? ???? ?????
        $fileName = 'external_' . md5($externalUrl . time()) . '.pdf';
        $filePath = getTargetPath($fileName, 'original');
        
        // ?? NEW: ????? ?????? ??? ?? ??? ???????
        $fileDir = dirname($filePath);
        if (!is_dir($fileDir)) {
            mkdir($fileDir, 0755, true);
            error_log("?? Created directory: $fileDir");
        }
        
        // ??? ?????
        $result = file_put_contents($filePath, $pdfData);
        
        if ($result === false) {
            error_log("? Failed to save file: $filePath");
            return false;
        }
        
        // ?????? ?? ?? ????? PDF ????
        if (!isPdfFileValid($filePath)) {
            unlink($filePath);
            error_log("? Downloaded file is not a valid PDF");
            return false;
        }
        
        $fileSize = filesize($filePath);
        error_log("? External PDF downloaded SUCCESS: $filePath, Size: $fileSize bytes");
        
        return $filePath;
        
    } catch (Exception $e) {
        error_log("? Download error: " . $e->getMessage());
        return false;
    }
}

// ?? NEW: ???? ??? PDF ?????
function isPdfFileValid($filePath) {
    try {
        if (!file_exists($filePath)) {
            error_log("? PDF validation: File not found - $filePath");
            return false;
        }
        
        $fileSize = filesize($filePath);
        if ($fileSize < 100) { // ??? PDF ?? ???? ?? ???? ??? ?? 100 ????
            error_log("? PDF validation: File too small - $fileSize bytes");
            return false;
        }
        
        $fileHandle = @fopen($filePath, 'r');
        if (!$fileHandle) {
            error_log("? PDF validation: Cannot open file - $filePath");
            return false;
        }
        
        $header = fread($fileHandle, 10); // ????? 10 ???? ??????
        fclose($fileHandle);
        
        $isValid = (strpos($header, '%PDF-') === 0);
        error_log("?? PDF validation: " . ($isValid ? "VALID" : "INVALID") . " - Header: " . bin2hex($header));
        
        return $isValid;
        
    } catch (Exception $e) {
        error_log("? PDF validation error: " . $e->getMessage());
        return false;
    }
}

// ?? NEW: ???? ?????? ?? ??? ???? PDF
function isValidPdfUrl($url) {
    try {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_NOBODY => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            CURLOPT_SSL_VERIFYPEER => false,
        ]);
        
        curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        $contentLength = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
        
        curl_close($ch);
        
        $isValid = ($httpCode == 200 && 
                   (strpos($contentType, 'application/pdf') !== false || 
                    strpos($contentType, 'application/octet-stream') !== false ||
                    preg_match('/\.pdf$/i', $url)) &&
                   $contentLength > 0);
        
        error_log("?? URL Validation: $url - HTTP: $httpCode, Type: $contentType, Size: $contentLength, Valid: " . ($isValid ? 'YES' : 'NO'));
        
        return $isValid;
        
    } catch (Exception $e) {
        error_log("? URL validation error: " . $e->getMessage());
        return false;
    }
}

// ?? NEW: ???? ????? ????? ????? PDF
function getPdfPageCount($pdfPath) {
    error_log("?? Counting pages for PDF: $pdfPath");
    
    if (!file_exists($pdfPath)) {
        error_log("? PDF file does not exist: $pdfPath");
        return 0;
    }
    
    $fileSize = filesize($pdfPath);
    error_log("?? PDF file size: $fileSize bytes");
    
    // ??????? Ghostscript ??????? ???????
    if ($fileSize > (5 * 1024 * 1024) && function_exists('shell_exec')) {
        $pageCount = getPdfPageCountGhostscript($pdfPath);
        if ($pageCount > 0) {
            error_log("? Using Ghostscript page count: $pageCount");
            return $pageCount;
        }
    }
    
    // ???????? ??????: ??????? FPDI
    $pageCount = getPdfPageCountFpdi($pdfPath);
    if ($pageCount > 0) {
        error_log("? Using FPDI page count: $pageCount");
        return $pageCount;
    }
    
    // ???????? ???????: ??????? shell command
    if (function_exists('shell_exec')) {
        $command = "pdfinfo \"$pdfPath\" 2>/dev/null | grep 'Pages:' | awk '{print \$2}'";
        $pageCount = shell_exec($command);
        
        if ($pageCount !== null && is_numeric(trim($pageCount))) {
            $count = intval(trim($pageCount));
            error_log("? PDF page count (pdfinfo): $count");
            return $count;
        }
    }
    
    // ???????? ???????: ??????? PHP preg_match
    $pageCount = countPdfPagesPhp($pdfPath);
    if ($pageCount > 0) {
        error_log("? Using PHP page count: $pageCount");
        return $pageCount;
    }
    
    // ???????? ???????: ??????? ??????? (??? ????)
    $estimatedPages = ceil($fileSize / (200 * 1024)); // 200KB per page
    error_log("?? Using estimated page count: $estimatedPages (File size: $fileSize bytes)");
    return $estimatedPages;
}

// ?? NEW: ???? ????? ????? ??????? ???????? Ghostscript
function getPdfPageCountGhostscript($pdfPath) {
    try {
        $command = "gs -q -dNODISPLAY -c \"($pdfPath) (r) file runpdfbegin pdfpagecount = quit\" 2>&1";
        $output = shell_exec($command);
        
        if (is_numeric(trim($output))) {
            $pageCount = intval(trim($output));
            error_log("?? Ghostscript page count: $pageCount");
            return $pageCount;
        }
        
        // ????? ????? ???????? Ghostscript
        $command = "gs -q -dNODISPLAY -c \"($pdfPath) (r) file runpdfbegin pdfpagecount == quit\" 2>&1";
        $output = shell_exec($command);
        
        if (is_numeric(trim($output))) {
            $pageCount = intval(trim($output));
            error_log("?? Ghostscript page count (alt): $pageCount");
            return $pageCount;
        }
        
        return 0;
    } catch (Exception $e) {
        error_log("? Ghostscript page count error: " . $e->getMessage());
        return 0;
    }
}

// ???? ?????: ???? ??? ??????? ???????? FPDI
function getPdfPageCountFpdi($pdfPath) {
    try {
        if (!file_exists($pdfPath)) {
            error_log("? PDF file does not exist: $pdfPath");
            return 0;
        }
        
        // ????? ???????? ?????? ??? ??? ?????
        if (!class_exists('FPDF')) {
            require_once __DIR__ . '/fpdf/fpdf.php';
            error_log("?? Loaded FPDF manually");
        }
        
        if (!class_exists('setasign\Fpdi\Fpdi')) {
            require_once __DIR__ . '/fpdi/src/autoload.php';
            error_log("?? Loaded FPDI manually");
        }
        
        $pdf = new \setasign\Fpdi\Fpdi();
        $pageCount = $pdf->setSourceFile($pdfPath);
        
        error_log("? FPDI Page count for $pdfPath: $pageCount");
        return $pageCount;
    } catch (Exception $e) {
        error_log("? FPDI page count error for $pdfPath: " . $e->getMessage());
        return 0;
    }
}

// ???? PHP ????? ????? ????? PDF
function countPdfPagesPhp($filename) {
    if (!file_exists($filename)) {
        error_log("? File not found: $filename");
        return 0;
    }
    
    try {
        $fp = @fopen($filename, "r");
        if (!$fp) {
            error_log("? Cannot open file: $filename");
            return 0;
        }
        
        $maxline = 1024;
        $count = 0;
        $content = '';
        
        while(!feof($fp)) {
            $line = fgets($fp, $maxline);
            $content .= $line;
            
            // ????? ?? ??? /Count ???? ???? ????? ???????? ???????
            if (preg_match_all("/\/Count\s+(\d+)/", $content, $matches)) {
                foreach ($matches[1] as $match) {
                    $currentCount = intval($match);
                    if ($currentCount > $count) {
                        $count = $currentCount;
                    }
                }
            }
            
            // ????? ?? ??? /Type/Page
            if (preg_match_all("/\/Type\s*\/Page[^a-zA-Z]/", $content, $matches)) {
                $currentCount = count($matches[0]);
                if ($currentCount > $count) {
                    $count = $currentCount;
                }
            }
            
            // ??? loop ?? ?????
            if (strlen($content) > 5000000) { // 5MB ??? ????
                error_log("?? Reached 5MB limit while reading PDF");
                break;
            }
        }
        
        fclose($fp);
        
        // ??? ?? ??? ????? ??????? ???? ?? ????? ?????
        if ($count === 0) {
            $content = file_get_contents($filename, false, null, 0, 1000000); // ??? 1MB ???
            if (preg_match("/\/Count\s+(\d+)/", $content, $matches)) {
                $count = intval($matches[1]);
                error_log("? Found page count in trailer: $count");
            } elseif (preg_match_all("/\/Type\s*\/Page[^a-zA-Z]/", $content, $matches)) {
                $count = count($matches[0]);
                error_log("? Found pages by Type/Page: $count");
            }
        }
        
        return $count;
        
    } catch (Exception $e) {
        error_log("? PHP PDF page count error: " . $e->getMessage());
        return 0;
    }
}

// ?? NEW: ???? ?????? ?????? ???????
function parsePageRanges($ranges) {
    error_log("?? Parsing page ranges: '$ranges'");
    
    $pages = [];
    $ranges = trim($ranges);
    
    if (empty($ranges)) {
        error_log("? Empty ranges string");
        return $pages;
    }
    
    // ????? ????????
    $rangeParts = explode(',', $ranges);
    error_log("?? Range parts: " . print_r($rangeParts, true));
    
    foreach ($rangeParts as $range) {
        $range = trim($range);
        if (empty($range)) continue;
        
        if (strpos($range, '-') !== false) {
            // ???? ?????
            $parts = explode('-', $range);
            if (count($parts) === 2) {
                $start = intval(trim($parts[0]));
                $end = intval(trim($parts[1]));
                
                if ($start > 0 && $end > 0 && $start <= $end) {
                    for ($i = $start; $i <= $end; $i++) {
                        $pages[] = $i;
                    }
                    error_log("? Added range: $start-$end");
                } else {
                    error_log("? Invalid range: $start-$end");
                }
            }
        } else {
            // ???? ?????
            $page = intval($range);
            if ($page > 0) {
                $pages[] = $page;
                error_log("? Added single page: $page");
            } else {
                error_log("? Invalid page: $page");
            }
        }
    }
    
    $pages = array_unique($pages);
    sort($pages);
    
    error_log("?? Final pages list: " . implode(', ', $pages));
    return $pages;
}

// ?? NEW: ???? ????? ?????? PDF ???????? Ghostscript
function extractWithGhostscript($sourcePdf, $outputPdf, $pageRanges) {
    error_log("?? Using Ghostscript for PDF extraction");
    
    // ????? ?????? ??????? ??? ????? Ghostscript
    $pages = parsePageRanges($pageRanges);
    
    if (empty($pages)) {
        error_log("? No valid pages found for Ghostscript extraction");
        return false;
    }
    
    $firstPage = min($pages);
    $lastPage = max($pages);
    
    error_log("?? Ghostscript pages: first=$firstPage, last=$lastPage, total pages: " . count($pages));
    
    // ???? ??? Ghostscript
    $command = "gs -q -dNOPAUSE -dBATCH -sDEVICE=pdfwrite -dPDFSETTINGS=/prepress " .
               "-dFirstPage=$firstPage -dLastPage=$lastPage " .
               "-sOutputFile=\"$outputPdf\" \"$sourcePdf\" 2>&1";
    
    error_log("? Ghostscript command: $command");
    
    // ????? ?????
    $output = shell_exec($command);
    
    // ?????? ?? ???????
    if (file_exists($outputPdf) && filesize($outputPdf) > 0) {
        $size = filesize($outputPdf);
        error_log("? Ghostscript extraction SUCCESS - File size: $size bytes");
        return true;
    } else {
        error_log("? Ghostscript extraction failed");
        error_log("Ghostscript output: " . ($output ? $output : 'No output'));
        
        // ???????? ?????? ?????
        return extractWithFpdi($sourcePdf, $outputPdf, $pageRanges);
    }
}

// ?? NEW: ???? ????? ?????? PDF ???????? FPDI
function extractWithFpdi($sourcePdf, $outputPdf, $pageRanges) {
    error_log("?? Using FPDI for PDF extraction");
    
    // ????? ???????? ?????? ??? ??? ?????
    if (!class_exists('FPDF')) {
        require_once __DIR__ . '/fpdf/fpdf.php';
        error_log("?? Loaded FPDF manually");
    }
    
    if (!class_exists('setasign\Fpdi\Fpdi')) {
        require_once __DIR__ . '/fpdi/src/autoload.php';
        error_log("?? Loaded FPDI manually");
    }
    
    // ????? ???? FPDI
    $pdf = new Fpdi();
    
    // ????? ??? ?????? ?? ?????? ???????
    try {
        $pageCount = $pdf->setSourceFile($sourcePdf);
        error_log("?? Total pages in source: $pageCount");
    } catch (Exception $e) {
        error_log("? FPDI setSourceFile error: " . $e->getMessage());
        return false;
    }
    
    // ????? ?????? ???????
    $pages = parsePageRanges($pageRanges);
    error_log("?? Pages to extract: " . implode(', ', $pages));
    
    if (empty($pages)) {
        error_log("? No valid pages found for FPDI extraction");
        return false;
    }
    
    // ????? ??????? ????????
    foreach ($pages as $pageNumber) {
        if ($pageNumber < 1 || $pageNumber > $pageCount) {
            error_log("?? Page $pageNumber is out of range (1-$pageCount) - skipping");
            continue;
        }
        
        try {
            $pdf->AddPage();
            $templateId = $pdf->importPage($pageNumber);
            $pdf->useTemplate($templateId);
            error_log("? Added page $pageNumber");
        } catch (Exception $e) {
            error_log("? Error adding page $pageNumber: " . $e->getMessage());
            continue;
        }
    }
    
    // ??? ????? ??????
    try {
        // ?????? ?? ???? ??????
        $outputDir = dirname($outputPdf);
        if (!is_dir($outputDir)) {
            mkdir($outputDir, 0755, true);
            error_log("? Created output directory: $outputDir");
        }
        
        $pdf->Output($outputPdf, 'F');
        
        // ?????? ?? ???????
        if (file_exists($outputPdf)) {
            $size = filesize($outputPdf);
            error_log("? FPDI extraction SUCCESS - File size: $size bytes");
            return true;
        } else {
            error_log("? Output file was not created: $outputPdf");
            return false;
        }
    } catch (Exception $e) {
        error_log("? FPDI output error: " . $e->getMessage());
        return false;
    }
}

// ?? NEW: ???? ????? ?????? PDF
function extractPdfPages($sourcePdf, $outputPdf, $pageRanges) {
    error_log("====== PDF EXTRACTION START ======");
    error_log("Source: $sourcePdf");
    error_log("Output: $outputPdf"); 
    error_log("Pages: '$pageRanges'");
    error_log("File size: " . (file_exists($sourcePdf) ? filesize($sourcePdf) : 0) . " bytes");
    
    // ?????? ?? ???????? ????????
    if (empty($pageRanges)) {
        error_log("? Empty page ranges - no extraction needed");
        return false;
    }
    
    if (!file_exists($sourcePdf)) {
        error_log("? Source file does not exist: $sourcePdf");
        return false;
    }
    
    // ????? ?????? ???????
    $pageRanges = trim($pageRanges);
    if (empty($pageRanges)) {
        error_log("? Empty page ranges after trim");
        return false;
    }
    
    // ?????? ?? ??? ????? - ??????? Ghostscript ??????? ???????
    $fileSize = filesize($sourcePdf);
    $useGhostscript = $fileSize > (5 * 1024 * 1024); // ??????? Ghostscript ??????? ???? ?? 5MB
    
    error_log("?? File size: $fileSize bytes, Use Ghostscript: " . ($useGhostscript ? 'YES' : 'NO'));
    
    try {
        if ($useGhostscript && function_exists('shell_exec')) {
            return extractWithGhostscript($sourcePdf, $outputPdf, $pageRanges);
        } else {
            return extractWithFpdi($sourcePdf, $outputPdf, $pageRanges);
        }
        
    } catch (Exception $e) {
        error_log("? EXTRACTION ERROR: " . $e->getMessage());
        
        // ???????? ???????? Ghostscript ??? ????
        if (function_exists('shell_exec')) {
            error_log("?? Trying Ghostscript as fallback...");
            return extractWithGhostscript($sourcePdf, $outputPdf, $pageRanges);
        }
        
        return false;
    }
}

// ?? NEW: ???? ????? ??????? ????? PDF
function processPdfExtraction($sourcePdf, $baseFileName, $selected_pages_original, $selected_pages_toc, $selected_pages_summary, &$original_pdf_url, &$pdf_url, &$file_size, &$extracted_pdf_url, &$extracted_file_size, &$extracted_summary_pdf_url, &$extracted_summary_file_size, $bookId, $pdo) {
    
    error_log("? Starting PDF extraction process with ORIGINAL FILE PRESERVATION...");
    
    // ?? NEW: ????? ???????? ????????
    createUploadDirectories();
    
    // ?????? 1: ??? ????? ?????? ?????? ?? ???? original
    $original_pdf_url = $sourcePdf;
    
    // ?? IMPORTANT: ??? ??? ????? ?????? ??? ?? ???? original? ????? ????
    $originalDir = getUploadDirectories()['original'];
    if (strpos($sourcePdf, $originalDir) === false) {
        $newOriginalPath = getTargetPath(basename($sourcePdf), 'original');
        if (rename($sourcePdf, $newOriginalPath)) {
            $original_pdf_url = $newOriginalPath;
            $sourcePdf = $newOriginalPath;
            error_log("? MOVED original PDF to permanent storage: $newOriginalPath");
        }
    }
    
    // ?????? 2: ?????? ??????? ??????? ??????? (Original PDF)
    $finalPdfUrl = $sourcePdf; // ?????????: ??????? ????? ??????
    $finalFileSize = filesize($sourcePdf);

    if (!empty($selected_pages_original)) {
        error_log("?? Extracting SPECIFIED PAGES for storage: '$selected_pages_original'");
        
        $extractedOriginalFileName = 'storage_' . $baseFileName;
        $extractedOriginalPdfPath = getTargetPath($extractedOriginalFileName, 'display');
        
        error_log("?? Extracting storage pages to: $extractedOriginalPdfPath");
        
        $originalExtractionResult = extractPdfPages($sourcePdf, $extractedOriginalPdfPath, $selected_pages_original);
        
        if ($originalExtractionResult && file_exists($extractedOriginalPdfPath)) {
            // ??????? ????? ??????? ???????
            $finalPdfUrl = $extractedOriginalPdfPath;
            $finalFileSize = filesize($extractedOriginalPdfPath);
            error_log("? Storage PDF created with selected pages");
        } else {
            error_log("? Storage PDF extraction failed - using original file");
            $finalPdfUrl = $sourcePdf;
            $finalFileSize = filesize($sourcePdf);
        }
    } else {
        error_log("?? No storage pages specified - using FULL original PDF");
        // ??? ?? ??? ????? ?????? ?????? ????? ?????? ??????
        $displayPath = getTargetPath(basename($sourcePdf), 'display');
        if ($sourcePdf !== $displayPath && copy($sourcePdf, $displayPath)) {
            $finalPdfUrl = $displayPath;
            $finalFileSize = filesize($displayPath);
            error_log("? Copied full original PDF to display folder: $displayPath");
        }
    }

    // ?? FIX: ????? ????????? ????????
    $pdf_url = $finalPdfUrl; // ????? ?? ???? display
    $file_size = $finalFileSize;

    // ?????? 3: ??????? TOC PDF ??? ???? ???? ????? ?????
    if (!empty($selected_pages_toc)) {
        error_log("?? Starting TOC PDF extraction...");
        
        $extractedTocFileName = 'toc_' . $baseFileName;
        $extractedTocPdfPath = getTargetPath($extractedTocFileName, 'extracted_toc');
        
        error_log("?? Extracting TOC pages '$selected_pages_toc' to: $extractedTocPdfPath");
        
        $tocExtractionResult = extractPdfPages($sourcePdf, $extractedTocPdfPath, $selected_pages_toc);
        
        if ($tocExtractionResult && file_exists($extractedTocPdfPath)) {
            $extracted_pdf_url = $extractedTocPdfPath;
            $extracted_file_size = filesize($extractedTocPdfPath);
            error_log("? TOC PDF extracted successfully");
        } else {
            $extracted_pdf_url = '';
            $extracted_file_size = 0;
            error_log("? TOC PDF extraction failed");
        }
    } else {
        $extracted_pdf_url = '';
        $extracted_file_size = 0;
        error_log("?? No TOC pages specified");
    }

    // ?????? 4: ??????? Summary PDF ??? ???? ???? ????? ?????
    if (!empty($selected_pages_summary)) {
        error_log("?? Starting SUMMARY PDF extraction...");
        
        $extractedSummaryFileName = 'summary_' . $baseFileName;
        $extractedSummaryPdfPath = getTargetPath($extractedSummaryFileName, 'extracted_summary');
        
        error_log("?? Extracting Summary pages '$selected_pages_summary' to: $extractedSummaryPdfPath");
        
        $summaryExtractionResult = extractPdfPages($sourcePdf, $extractedSummaryPdfPath, $selected_pages_summary);
        
        if ($summaryExtractionResult && file_exists($extractedSummaryPdfPath)) {
            $extracted_summary_pdf_url = $extractedSummaryPdfPath;
            $extracted_summary_file_size = filesize($extractedSummaryPdfPath);
            error_log("? Summary PDF extracted successfully");
        } else {
            $extracted_summary_pdf_url = '';
            $extracted_summary_file_size = 0;
            error_log("? Summary PDF extraction failed");
        }
    } else {
        $extracted_summary_pdf_url = '';
        $extracted_summary_file_size = 0;
        error_log("?? No Summary pages specified");
    }
    
    error_log("? PDF extraction process completed successfully");
    error_log("=== FINAL FILE STRUCTURE ===");
    error_log("Original PDF (ALWAYS PRESERVED): $original_pdf_url");
    error_log("Display PDF (for reading): $pdf_url");
    error_log("TOC PDF: " . ($extracted_pdf_url ?: 'NOT EXTRACTED'));
    error_log("Summary PDF: " . ($extracted_summary_pdf_url ?: 'NOT EXTRACTED'));
    
    return true;
}

// ?? NEW: ???? ?????? ??????? ???????? ???????
function cleanupOldExtractedFiles($pdo, $bookId) {
    error_log("?? Cleaning old EXTRACTED files (keeping original) for book ID: $bookId");
    
    $stmt = $pdo->prepare('SELECT pdf_url, original_pdf_url, extracted_pdf_url, extracted_summary_pdf_url FROM books WHERE id = ?');
    $stmt->execute([$bookId]);
    $oldBook = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($oldBook) {
        $filesToDelete = [];
        
        // ????? ??????? ???????? ????? (?? ???? ????? ??????)
        $extractedFiles = [
            $oldBook['pdf_url'], // ?? ???? display
            $oldBook['extracted_pdf_url'], // ?? ???? extracted_toc
            $oldBook['extracted_summary_pdf_url'] // ?? ???? extracted_summary
        ];
        
        foreach ($extractedFiles as $filePath) {
            if (!empty($filePath) && $filePath !== 'uploads/no-image.jpg') {
                $filesToDelete[] = $filePath;
            }
        }
        
        // ?? IMPORTANT: ?? ???? ????? ?????? ?? ???? original ?????
        
        $deletedCount = 0;
        foreach ($filesToDelete as $filePath) {
            if (!empty($filePath) && 
                file_exists($filePath) && 
                !filter_var($filePath, FILTER_VALIDATE_URL) &&
                $filePath !== 'uploads/no-image.jpg' &&
                $filePath !== '../api/uploads/no-image.jpg') {
                
                if (unlink($filePath)) {
                    error_log("? Deleted extracted file: " . basename($filePath));
                    $deletedCount++;
                } else {
                    error_log("? Failed to delete file: $filePath");
                }
            }
        }
        
        error_log("?? Deleted $deletedCount extracted files (ORIGINAL FILE PRESERVED)");
    }
}

// ?? NEW: Handle download count increment
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['increment_download']) && $_POST['increment_download'] === 'true') {
    $bookId = $_POST['book_id'] ?? null;
    
    if ($bookId) {
        try {
            $stmt = $pdo->prepare('UPDATE books SET download_count = download_count + 1 WHERE id = ?');
            $stmt->execute([$bookId]);
            
            error_log("? Download count incremented for book ID: $bookId");
            json_success('Download count updated');
        } catch (PDOException $e) {
            error_log("? Error incrementing download count: " . $e->getMessage());
            http_response_code(500);
            json_error('Failed to update download count');
        }
    } else {
        http_response_code(400);
        json_error('Book ID required');
    }
}

// GET: List books - ???? ???? ??????? ??? ??????
if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    try {
        error_log("=== GET Request for books ===");
        
        $categoryId = $_GET['category_id'] ?? null;
        $profileCategoryId = $_GET['profile_category_id'] ?? null;
        $activeOnly = isset($_GET['active_only']) && $_GET['active_only'] === 'true';

        // ???? ????????? ?? ??? ??????? ??? ??????
        $whereConditions = [];
        $params = [];

        if ($activeOnly) {
            $whereConditions[] = "b.is_active = TRUE";
            error_log("?? ??? ????? ?????? ???");
        }

        if ($categoryId) {
            $whereConditions[] = "b.category_id = ?";
            $params[] = $categoryId;
        } elseif ($profileCategoryId) {
            $whereConditions[] = "b.profile_category_id = ?";
            $params[] = $profileCategoryId;
        }

        $whereClause = $whereConditions ? "WHERE " . implode(" AND ", $whereConditions) : "";

        $sql = "SELECT b.*, c.name as category_name, pc.name as profile_category_name 
                FROM books b 
                LEFT JOIN categories c ON b.category_id = c.id 
                LEFT JOIN profile_categories pc ON b.profile_category_id = pc.id 
                $whereClause 
                ORDER BY b.created_at DESC";

        error_log("?? SQL Query: $sql");
        error_log("?? Parameters: " . print_r($params, true));

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $books = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $response = [];

        foreach ($books as $book) {
            $isExternal = $book['is_external'] ?? 0;
            $isExternalPlus = $book['is_external_plus'] ?? 0;
            $fileSize = 0;
            $extractedFileSize = 0;
            $extractedSummaryFileSize = 0;
            $pageCount = 0;
            
            // ?? ??????? ???????: ??????? original_pdf_url ????? ?? pdf_url
            $pdfSourceForPageCount = !empty($book['original_pdf_url']) ? $book['original_pdf_url'] : $book['pdf_url'];
            
            if ($isExternal && filter_var($pdfSourceForPageCount, FILTER_VALIDATE_URL)) {
                // External URL - ?? ???? ???? ??????? ????
                $fileSize = getRemoteFileSize($pdfSourceForPageCount);
                $pageCount = $book['page_count'] ?? 0; // ??????? ?????? ???????
                
                if (!empty($book['extracted_pdf_url']) && filter_var($book['extracted_pdf_url'], FILTER_VALIDATE_URL)) {
                    $extractedFileSize = getRemoteFileSize($book['extracted_pdf_url']);
                }
                if (!empty($book['extracted_summary_pdf_url']) && filter_var($book['extracted_summary_pdf_url'], FILTER_VALIDATE_URL)) {
                    $extractedSummaryFileSize = getRemoteFileSize($book['extracted_summary_pdf_url']);
                }
            } else {
                // Local files - ???? ??????? ??????? ?? original_pdf_url
                if (!empty($pdfSourceForPageCount) && file_exists($pdfSourceForPageCount)) {
                    $fileSize = filesize($pdfSourceForPageCount);
                    
                    // ?? ??????? ???????: ???? ??????? ?? original_pdf_url
                    $pageCount = getPdfPageCount($pdfSourceForPageCount);
                    
                    // ??? ?? ????? ?? ???? ???????? ?????? ?????? ???????
                    if ($pageCount === 0 && isset($book['page_count'])) {
                        $pageCount = $book['page_count'];
                    }
                    
                    // ????? ????? ???????? ??????? ???????
                    if ($pageCount > 0 && $pageCount != $book['page_count']) {
                        $updateStmt = $pdo->prepare('UPDATE books SET page_count = ? WHERE id = ?');
                        $updateStmt->execute([$pageCount, $book['id']]);
                        error_log("?? Updated page count for book {$book['id']}: {$book['page_count']} -> $pageCount");
                    }
                }
                
                if (!empty($book['extracted_pdf_url']) && file_exists($book['extracted_pdf_url'])) {
                    $extractedFileSize = filesize($book['extracted_pdf_url']);
                }
                if (!empty($book['extracted_summary_pdf_url']) && file_exists($book['extracted_summary_pdf_url'])) {
                    $extractedSummaryFileSize = filesize($book['extracted_summary_pdf_url']);
                }
            }

            // ??? ?? ????? ?? ???? ???????? ?????? ?????? ???????
            if ($pageCount === 0 && isset($book['page_count'])) {
                $pageCount = $book['page_count'];
            }

            // Debug log for fields
            error_log("?? Book: {$book['title']}");
            error_log("   File Size: $fileSize bytes");
            error_log("   Page Count: $pageCount pages");
            error_log("   PDF Source for Page Count: $pdfSourceForPageCount");
            error_log("   Is External: $isExternal");
            error_log("   Is External Plus: $isExternalPlus");
            error_log("   Download Count: {$book['download_count']}");
            error_log("   Language: {$book['language']}");
            error_log("   Format: {$book['format']}");
            error_log("   External Download Link: {$book['external_download_link']}");
            error_log("   Show External Download Button: {$book['show_external_download_button']}");
            error_log("   ?? NEW: Is Active: {$book['is_active']}");
            error_log("   ?? NEW: Require Login for Download: {$book['require_login_for_download']}");
            error_log("   ?? NEW: External Download Protection Key: {$book['external_download_protection_key']}");

            $response[] = [
                'id' => $book['id'],
                'title' => $book['title'],
                'author' => $book['author'] ?? '',
                'category_id' => $book['category_id'],
                'profile_category_id' => $book['profile_category_id'] ?? null,
                'category_name' => $book['category_name'] ?? null,
                'profile_category_name' => $book['profile_category_name'] ?? null,
                'image_url' => $book['image_url'],
                'pdf_url' => $book['pdf_url'],
                'original_pdf_url' => $book['original_pdf_url'] ?? $book['pdf_url'],
                'extracted_pdf_url' => $book['extracted_pdf_url'] ?? '',
                'extracted_summary_pdf_url' => $book['extracted_summary_pdf_url'] ?? '',
                'file_size' => $fileSize,
                'extracted_file_size' => $extractedFileSize,
                'extracted_summary_file_size' => $extractedSummaryFileSize,
                'page_count' => $pageCount,
                'summary' => $book['summary'],
                'is_external' => $isExternal,
                'is_external_plus' => $isExternalPlus,
                'table_of_contents' => $book['table_of_contents'] ?? '',
                'selected_pages_summary' => $book['selected_pages_summary'] ?? '',
                'selected_pages_toc' => $book['selected_pages_toc'] ?? '',
                'selected_pages_original' => $book['selected_pages_original'] ?? '',
                'institution' => $book['institution'] ?? '',
                'publication_year' => $book['publication_year'] ?? '',
                'keywords' => $book['keywords'] ?? '',
                'language' => $book['language'] ?? '',
                'format' => $book['format'] ?? '',
                
                // ?????? ??????? ???????
                'is_protected' => (bool)($book['is_protected'] ?? false),
                'protection_key' => $book['protection_key'] ?? '',
                
                // ?????? ??????? ?????? ?? ?? ???????
                'show_read_button' => (bool)($book['show_read_button'] ?? true),
                'read_button_text' => $book['read_button_text'] ?? 'Lire le document',
                
                // ?????? ??????? ?????? ?? ??? ?????
                'custom_file_size' => $book['custom_file_size'] ?? '',
                'show_file_size' => (bool)($book['show_file_size'] ?? true),
                
                // ????? ??????: Taille befor Extrait
                'taille_before_extrait' => $book['taille_before_extrait'] ?? '',
                'show_taille_before_extrait' => (bool)($book['show_taille_before_extrait'] ?? true),
                
                // NEW: Download Extrait button control fields
                'show_download_extrait_button' => (bool)($book['show_download_extrait_button'] ?? true),
                'download_extrait_button_text' => $book['download_extrait_button_text'] ?? 'Download extrait',
                
                // ?? NEW: Download counter
                'download_count' => (int)($book['download_count'] ?? 0),
                
                // ?? NEW: Stockage field
                'stockage' => $book['stockage'] ?? '',

                // ?? NEW: External download link fields
                'external_download_link' => $book['external_download_link'] ?? '',
                'show_external_download_button' => (bool)($book['show_external_download_button'] ?? false),
                
                // ?? NEW: External download protection fields
                'external_download_protection_key' => $book['external_download_protection_key'] ?? '',
                'is_external_download_protected' => (bool)($book['is_external_download_protected'] ?? false),
				
				// 🔥 NEW: Require login for download field
                'require_login_for_download' => (bool)($book['require_login_for_download'] ?? false),

                // ?? NEW: Active status field
                'is_active' => (bool)($book['is_active'] ?? true),
            ];
        }

        error_log("? Returning " . count($response) . " books");
        clean_json_output($response);

    } catch (PDOException $e) {
        error_log("? Database error: " . $e->getMessage());
        http_response_code(500);
        json_error('Database error: ' . $e->getMessage());
    }
}

// POST: Add or Update book - ????? ?????? ???????
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        error_log("=== BOOKS.PHP POST REQUEST ===");
        error_log("POST data: " . print_r($_POST, true));
        error_log("FILES data: " . print_r($_FILES, true));
        
        // ?? NEW: ????? ????? ??? ????????
        ob_clean();
        
        // ?? NEW: ????? ???????? ???????? ?????
        createUploadDirectories();
        
        $id = $_POST['id'] ?? null;
        $title = $_POST['title'] ?? '';
        $author = $_POST['author'] ?? null;
        $categoryId = $_POST['category_id'] ?? null;
        $profileCategoryId = $_POST['profile_category_id'] ?? null;
        $summary = $_POST['summary'] ?? '';
        $table_of_contents = $_POST['table_of_contents'] ?? '';
        
        // ?? NEW: Language Field
        $language = $_POST['language'] ?? '';
        
        // ?? NEW: Format Field
        $format = $_POST['format'] ?? '';
        
        // ?? NEW: Page Count Field
        $page_count_input = $_POST['page_count'] ?? null;
        $page_count = $page_count_input !== null ? intval($page_count_input) : null;
        
        // ?? NEW: External Download Link Fields
        $external_download_link = $_POST['external_download_link'] ?? '';
        $show_external_download_button = isset($_POST['show_external_download_button']) && $_POST['show_external_download_button'] === 'true' ? 1 : 0;
        
        // ?? NEW: External Download Protection Fields
        $external_download_protection_key = $_POST['external_download_protection_key'] ?? '';
        $is_external_download_protected = isset($_POST['is_external_download_protected']) && $_POST['is_external_download_protected'] === 'true' ? 1 : 0;
        
        // ?? FIX CRITICAL: ????? ??????? ??? ??????
        $isActive = isset($_POST['is_active']) ? ($_POST['is_active'] === 'true' ? 1 : 0) : 1;
        $require_login_for_download = isset($_POST['require_login_for_download']) ? ($_POST['require_login_for_download'] === 'true' ? 1 : 0) : 0;
        
        // ?? DEBUG: ????? ????? ????????
        error_log("?? SERVER - Received checkbox values:");
        error_log("   is_active from POST: " . ($_POST['is_active'] ?? 'NOT SET'));
        error_log("   require_login_for_download from POST: " . ($_POST['require_login_for_download'] ?? 'NOT SET'));
        error_log("   Parsed is_active: " . $isActive);
        error_log("   Parsed require_login_for_download: " . $require_login_for_download);
        
        error_log("?? Language from form: $language");
        error_log("?? Format from form: $format");
        error_log("?? Page Count from form: " . ($page_count !== null ? $page_count : 'NULL (auto-calculate)'));
        error_log("?? External Download Link: $external_download_link");
        error_log("?? Show External Download Button: $show_external_download_button");
        error_log("?? External Download Protection Key: $external_download_protection_key");
        error_log("?? Is External Download Protected: $is_external_download_protected");
        error_log("?? Is Active: $isActive");
        error_log("?? Require Login for Download: $require_login_for_download");
        
        // ?? NEW: Default image flag
        $use_default_image = isset($_POST['use_default_image']) && $_POST['use_default_image'] === 'true';
        
        // ???? ???????
        $selected_pages_summary = $_POST['selected_pages_summary'] ?? '';
        $selected_pages_toc = $_POST['selected_pages_toc'] ?? '';
        $selected_pages_original = $_POST['selected_pages_original'] ?? '';
        
        $pdf_url_option = $_POST['pdf_url_option'] ?? 'file';
        $external_pdf_url = $_POST['external_pdf_url'] ?? '';
        $external_pdf_url_plus = $_POST['external_pdf_url_plus'] ?? '';
        
        // ?????? ??????
        $institution = $_POST['institution'] ?? '';
        $publication_year = $_POST['publication_year'] ?? '';
        $keywords = $_POST['keywords'] ?? '';
        
        // ?? ??????? ??????: ?????? ??? ??????? ???? ????
        $is_protected = isset($_POST['is_protected']) && $_POST['is_protected'] === 'true' ? 1 : 0;
        $protection_key = $_POST['protection_key'] ?? '';

        // ??? ???? ??????? ??? ?????? ???? ?? ?? ????? ??????? ????
        if (!$is_protected) {
            $protection_key = '';
        }
        
        $show_read_button = isset($_POST['show_read_button']) && $_POST['show_read_button'] === 'true' ? 1 : 0;
        $read_button_text = $_POST['read_button_text'] ?? 'Lire le document';
        
        // ?????? ??????? ?????? ?? ??? ?????
        $custom_file_size = $_POST['custom_file_size'] ?? '';
        $show_file_size = isset($_POST['show_file_size']) && $_POST['show_file_size'] === 'true' ? 1 : 0;
        
        // ?????? ??????? ?? Taille befor Extrait
        $taille_before_extrait = $_POST['taille_before_extrait'] ?? '';
        $show_taille_before_extrait = isset($_POST['show_taille_before_extrait']) && $_POST['show_taille_before_extrait'] === 'true' ? 1 : 0;

        // NEW: Download Extrait button fields
        $show_download_extrait_button = isset($_POST['show_download_extrait_button']) && $_POST['show_download_extrait_button'] === 'true' ? 1 : 0;
        $download_extrait_button_text = $_POST['download_extrait_button_text'] ?? 'Download extrait';
        
        // ?? NEW: Stockage field
        $stockage = $_POST['stockage'] ?? '';
        
        // Remove image flag
        $remove_image = isset($_POST['remove_image']) && $_POST['remove_image'] === 'true';
        
        // ????????? ????????
        $image_url = '';
        $original_pdf_url = '';
        $pdf_url = '';
        $extracted_pdf_url = '';
        $extracted_summary_pdf_url = '';
        $file_size = 0;
        $extracted_file_size = 0;
        $extracted_summary_file_size = 0;
        $is_external = 0;
        $is_external_plus = 0;
        $calculated_page_count = 0;
        $page_count_updated = false;

        error_log("?? Processing book ID: $id");
        error_log("?? Language: $language");
        error_log("?? Format: $format");
        error_log("?? Page Count Strategy: " . ($page_count !== null ? "Manual: $page_count" : "Auto-calculate"));
        error_log("??? Remove image flag: " . ($remove_image ? 'YES' : 'NO'));
        error_log("??? Use default image flag: " . ($use_default_image ? 'YES' : 'NO'));
        error_log("??? Image file present: " . (isset($_FILES['image']) ? 'YES' : 'NO'));
        error_log("?? PDF Source Option: $pdf_url_option");
        error_log("?? Stockage: $stockage");
        error_log("?? External Download Link: $external_download_link");
        error_log("?? Show External Download Button: $show_external_download_button");
        error_log("?? External Download Protection Key: $external_download_protection_key");
        error_log("?? Is External Download Protected: $is_external_download_protected");
        error_log("??? PROTECTION STATUS - is_protected: $is_protected, protection_key: $protection_key");
        error_log("??? CRITICAL FIELDS - is_active: $isActive, require_login_for_download: $require_login_for_download");

        // NEW: Debug for download extrait button
        error_log("?? DOWNLOAD EXTRAIT BUTTON DATA:");
        error_log("   show_download_extrait_button: $show_download_extrait_button");
        error_log("   download_extrait_button_text: $download_extrait_button_text");

        // ?????? ?? ???????? ????????
        if (empty($title)) {
            error_log("? Validation failed: Title is required");
            json_error('Title is required');
        }

        if (empty($categoryId) && empty($profileCategoryId)) {
            error_log("? Validation failed: No category selected");
            json_error('Please select at least one category');
        }

        // ?? NEW: ?????? ?? ????? ???????
        error_log("?? Checking for duplicate book title: '$title'");
        if ($id) {
            // ???? ???????: ?????? ?? ??????? ?? ??????? ?????? ??????
            $checkStmt = $pdo->prepare('SELECT id, title FROM books WHERE title = ? AND id != ?');
            $checkStmt->execute([$title, $id]);
        } else {
            // ???? ???????: ?????? ?? ??????? ???? ???????
            $checkStmt = $pdo->prepare('SELECT id, title FROM books WHERE title = ?');
            $checkStmt->execute([$title]);
        }

        $existingBook = $checkStmt->fetch(PDO::FETCH_ASSOC);

        if ($existingBook) {
            $errorMessage = "A book with the same title ('{$existingBook['title']}') Already exists. Please use a different title.";
            error_log("? Validation failed: Book with same title already exists - ID: {$existingBook['id']}, Title: {$existingBook['title']}");
            json_error($errorMessage);
        }
        error_log("? No duplicate book title found");

        // Handle image removal
        if ($id && $remove_image) {
            error_log("??? Removing existing book image...");
            $stmt = $pdo->prepare('SELECT image_url FROM books WHERE id = ?');
            $stmt->execute([$id]);
            $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingBook && !empty($existingBook['image_url']) && file_exists($existingBook['image_url'])) {
                if (unlink($existingBook['image_url'])) {
                    error_log("? Deleted existing book image: " . $existingBook['image_url']);
                }
            }
            $image_url = ''; // Set empty image URL
            error_log("? Image URL set to empty after removal");
        }

        // ?? NEW: Auto-set default image for new books
        if (!$remove_image && !isset($_FILES['image']) && $use_default_image) {
            $defaultImagePath = 'uploads/no-image.jpg';
            $image_url = $defaultImagePath;
            error_log("? Using default image: $defaultImagePath");
        }

        // ?? NEW: ??? ??????? ??????? ??? ?? ??????
        if ($id) {
            cleanupOldExtractedFiles($pdo, $id);
        }

        // ?? NEW: ???????? ??????? ??????? ????????
        $uploadDir = getUploadDirectories()['original'];

        // ?? NEW: ????? ??????? ????? ???????? ??? ????? ???????
        if ($id && !isset($_FILES['pdf'])) {
            error_log("?? No new PDF uploaded - checking for page extraction changes");
            
            // ??? ??????? ?????? ??????
            $stmt = $pdo->prepare('SELECT pdf_url, original_pdf_url, file_size, selected_pages_summary, selected_pages_toc, selected_pages_original FROM books WHERE id = ?');
            $stmt->execute([$id]);
            $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingBook) {
                // ?? IMPORTANT: ?????? ?????? ????? ?????? ?????? ???????
                $originalPdfPath = $existingBook['original_pdf_url'] ?? $existingBook['pdf_url'];
                
                // ???????: ?????? ??? ????? ????? ???????
                $pagesChanged = (
                    $selected_pages_summary !== $existingBook['selected_pages_summary'] ||
                    $selected_pages_toc !== $existingBook['selected_pages_toc'] ||
                    $selected_pages_original !== $existingBook['selected_pages_original']
                );
                
                // ???????: ?? ??? ???? ??????? ????? ?????? ?????? ????? ????
                $pagesBecameSpecified = (
                    (empty($existingBook['selected_pages_summary']) && !empty($selected_pages_summary)) ||
                    (empty($existingBook['selected_pages_toc']) && !empty($selected_pages_toc)) ||
                    (empty($existingBook['selected_pages_original']) && !empty($selected_pages_original))
                );
                
                // ?? FIX CRITICAL: ????? ??????? ?????? ??? ???? ???? ????? ?????
                $hasPagesSpecified = !empty($selected_pages_summary) || !empty($selected_pages_toc) || !empty($selected_pages_original);
                
                if ($pagesChanged || $pagesBecameSpecified || $hasPagesSpecified) {
                    error_log("?? Pages changed/specified - RE-EXTRACTING from ORIGINAL: $originalPdfPath");
                    error_log("?? Extraction details:");
                    error_log("   Summary: '$selected_pages_summary'");
                    error_log("   TOC: '$selected_pages_toc'");
                    error_log("   Storage: '$selected_pages_original'");
                    
                    // ????? ??????? ???????? ??????? ?????
                    cleanupOldExtractedFiles($pdo, $id);
                    
                    // ????? ??????? ?? ????? ??????
                    processPdfExtraction($originalPdfPath, basename($originalPdfPath), 
                                        $selected_pages_original, $selected_pages_toc, $selected_pages_summary, 
                                        $original_pdf_url, $pdf_url, $file_size, $extracted_pdf_url, 
                                        $extracted_file_size, $extracted_summary_pdf_url, 
                                        $extracted_summary_file_size, $id, $pdo);
                } else {
                    // ??? ?? ????? ???????? ?????? ??????? ???????
                    $pdf_url = $existingBook['pdf_url'];
                    $original_pdf_url = $existingBook['original_pdf_url'] ?? $existingBook['pdf_url'];
                    $file_size = $existingBook['file_size'];
                    error_log("?? Using existing PDF files - no page changes detected");
                }
            }
        }

        // Handle PDF source - ????? ?????? ??????? ????????
        if ($pdf_url_option === 'url') {
            // ?? NEW: Regular External Link - ???????? ???????
            error_log("???? Processing regular external PDF URL WITH ENHANCED PAGE COUNT");
            
            if (empty($external_pdf_url)) {
                error_log("? External PDF URL is required");
                json_error('External PDF URL is required');
            }

            if (!filter_var($external_pdf_url, FILTER_VALIDATE_URL)) {
                error_log("? Invalid URL format");
                json_error('Invalid URL format');
            }

            // ????? ????? ???????
            error_log("?? DOWNLOADING EXTERNAL PDF: $external_pdf_url");
            $localPdfPath = downloadExternalPdf($external_pdf_url, $uploadDir);
            
            if (!$localPdfPath) {
                error_log("? Failed to download external PDF");
                json_error('Failed to download external PDF file');
            }

            // ??? ????? ??????
            if (file_exists($localPdfPath)) {
                $fileSize = filesize($localPdfPath);
                error_log("? External PDF downloaded - Path: $localPdfPath, Size: $fileSize bytes");
                
                // ??? ??? ??? ????? PDF ????
                if (isPdfFileValid($localPdfPath)) {
                    error_log("? Downloaded file is valid PDF");
                } else {
                    error_log("? Downloaded file is NOT a valid PDF");
                }
            } else {
                error_log("? Downloaded file doesn't exist: $localPdfPath");
            }

            // ??????? ????? ?????? ????? ?? ?????? ???????
            $originalPdfPath = $localPdfPath;
            $original_pdf_url = $localPdfPath;
            $pdf_url = $localPdfPath;
            $file_size = filesize($localPdfPath);
            $is_external = 1;
            $is_external_plus = 0;

            // ?? NEW: ???? ??????? ???? ????? ??????? ???????
            error_log("?? STARTING PAGE COUNT CALCULATION FOR EXTERNAL PDF");
            $calculated_page_count = getPdfPageCount($localPdfPath);
            error_log("?? EXTERNAL PDF PAGE COUNT RESULT: $calculated_page_count");
            
            // ?? NEW: ??? ??? ???? ???????? ???? ?????? ?????
            if ($calculated_page_count === 0 || $calculated_page_count === null) {
                error_log("?? Trying alternative page count methods...");
                
                // ???????? ??????: FPDI ??????
                $fpdi_count = getPdfPageCountFpdi($localPdfPath);
                error_log("?? FPDI Page count: $fpdi_count");
                
                // ???????? ???????: Ghostscript ??????
                $gs_count = getPdfPageCountGhostscript($localPdfPath);
                error_log("?? Ghostscript Page count: $gs_count");
                
                // ???????? ???????: PHP method
                $php_count = countPdfPagesPhp($localPdfPath);
                error_log("?? PHP Page count: $php_count");
                
                // ??????? ???? ?????
                $calculated_page_count = max($fpdi_count, $gs_count, $php_count);
                error_log("?? FINAL EXTERNAL PDF PAGE COUNT: $calculated_page_count");
            }

            // ?????? ??????? ???? ????? ??????? ???????
            error_log("?? Starting PDF extraction for external file");
            processPdfExtraction($originalPdfPath, basename($localPdfPath), $selected_pages_original, $selected_pages_toc, $selected_pages_summary, $original_pdf_url, $pdf_url, $file_size, $extracted_pdf_url, $extracted_file_size, $extracted_summary_pdf_url, $extracted_summary_file_size, $id, $pdo);

            error_log("? External PDF processing COMPLETED");

        } elseif ($pdf_url_option === 'url_plus') {
            // External Link Plus - Direct link without downloading
            error_log("?? Processing External Link Plus - Direct URL access");
            
            if (empty($external_pdf_url_plus)) {
                error_log("? External Link Plus URL is required");
                json_error('External Link Plus URL is required');
            }

            if (!filter_var($external_pdf_url_plus, FILTER_VALIDATE_URL)) {
                error_log("? Invalid URL format for External Link Plus");
                json_error('Invalid URL format for External Link Plus');
            }

            // ?????? ?? ?? ?????? ???? ??? ??? PDF
            if (!preg_match('/\.pdf$/i', $external_pdf_url_plus) && 
                !isValidPdfUrl($external_pdf_url_plus)) {
                error_log("? URL does not point to a valid PDF file");
                json_error('URL does not point to a valid PDF file');
            }

            // ??????? ?????? ?????? ???? ?????
            $pdf_url = $external_pdf_url_plus;
            $original_pdf_url = $external_pdf_url_plus;
            $is_external = 1;
            $is_external_plus = 1;
            
            // ?? ???? ???? ??? ????? ?? ??? ??????? ??????? ????????
            $file_size = 0;
            $calculated_page_count = 0;
            
            // ????? ???? ??????? ???????? ??? ??????? ??? ?????
            $extracted_pdf_url = '';
            $extracted_summary_pdf_url = '';
            $extracted_file_size = 0;
            $extracted_summary_file_size = 0;
            
            error_log("? External Link Plus configured - Direct URL: $external_pdf_url_plus");
            error_log("?? Page extraction is not supported for External Link Plus. URLs will be used directly.");
            
        } elseif ($pdf_url_option === 'file' && isset($_FILES['pdf']) && $_FILES['pdf']['error'] == 0) {
            // File upload - ????? ??????
            error_log("?? Processing NEW file upload");
            
            $uploadDir = getUploadDirectories()['original'];
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
                error_log("? Created upload directory: $uploadDir");
            }

            if ($_FILES['pdf']['type'] != 'application/pdf') {
                error_log("? Invalid file type: " . $_FILES['pdf']['type']);
                json_error('PDF file required');
            }

            $ext = pathinfo($_FILES['pdf']['name'], PATHINFO_EXTENSION);
            $fileName = uniqid() . '_' . time() . '.' . $ext;
            $targetFile = getTargetPath($fileName, 'original');

            error_log("?? Uploading file to: $targetFile");
            error_log("?? File size: " . $_FILES['pdf']['size'] . " bytes");

            if (move_uploaded_file($_FILES['pdf']['tmp_name'], $targetFile)) {
                $originalPdfPath = $targetFile;
                $original_pdf_url = $targetFile;
                error_log("?? Original PDF path saved: $originalPdfPath");
                
                // ???? ??????? ????? ??????
                $calculated_page_count = getPdfPageCount($originalPdfPath);
                error_log("?? Local PDF page count calculated: $calculated_page_count");
                
                // ?????? ??????? ????? ??????
                processPdfExtraction($originalPdfPath, $fileName, $selected_pages_original, $selected_pages_toc, $selected_pages_summary, $original_pdf_url, $pdf_url, $file_size, $extracted_pdf_url, $extracted_file_size, $extracted_summary_pdf_url, $extracted_summary_file_size, $id, $pdo);
                
            } else {
                error_log("? Failed to upload PDF");
                json_error('Failed to upload PDF');
            }
        } elseif ($id && !isset($_FILES['pdf'])) {
            error_log("?? No PDF changes detected, keeping existing files");
            // ?? ????? ?? ???????? ??????? ????? ???????
            $stmt = $pdo->prepare('SELECT pdf_url, original_pdf_url, extracted_pdf_url, extracted_summary_pdf_url, file_size, extracted_file_size, extracted_summary_file_size, page_count, is_external_plus FROM books WHERE id = ?');
            $stmt->execute([$id]);
            $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingBook) {
                $pdf_url = $existingBook['pdf_url'];
                $original_pdf_url = $existingBook['original_pdf_url'] ?? $existingBook['pdf_url'];
                $extracted_pdf_url = $existingBook['extracted_pdf_url'];
                $extracted_summary_pdf_url = $existingBook['extracted_summary_pdf_url'];
                $file_size = $existingBook['file_size'];
                $extracted_file_size = $existingBook['extracted_file_size'];
                $extracted_summary_file_size = $existingBook['extracted_summary_file_size'];
                $calculated_page_count = $existingBook['page_count'];
                $is_external_plus = $existingBook['is_external_plus'] ?? 0;
                error_log("?? Using existing PDF files without changes");
                
                // ????? ???? ??????? ??? ??? ?????
                if (($calculated_page_count === 0 || $calculated_page_count === null) && !empty($original_pdf_url) && file_exists($original_pdf_url)) {
                    $calculated_page_count = getPdfPageCount($original_pdf_url);
                    error_log("?? Recalculated page count from original PDF: $calculated_page_count");
                }
            }
        } elseif (!$id) {
            error_log("? PDF file is required for new books");
            json_error('PDF file is required for new books');
        }

        // ?? NEW: ???? ???? ??? ??????? ??????? - ????
$final_page_count = $page_count; // ?????? ?????????? ?? ?????? ???????
$page_count_updated = false;

// ?? FIX CRITICAL: ????? ???????? ?????? ??????? ??????
if ($page_count !== null && $page_count !== '') {
    // ??? ?? ????? ???? ??????? ???????? ??? ????? ?? ?? ???
    $final_page_count = intval($page_count);
    $page_count_updated = true;
    error_log("?? USING MANUAL PAGE COUNT: $final_page_count (User input)");
} elseif (!$id && !empty($originalPdfPath) && file_exists($originalPdfPath)) {
    // ??? ??? ???? ???? ??? ??? ????? ????? ???? ????????
    $final_page_count = $calculated_page_count;
    $page_count_updated = false;
    error_log("?? Auto-calculated page count for new book: $calculated_page_count");
} elseif ($id && $page_count === null) {
    // ??? ??? ????? ??? ??? ????? ????? ????? ??????? ???????
    $stmt = $pdo->prepare('SELECT page_count FROM books WHERE id = ?');
    $stmt->execute([$id]);
    $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($existingBook && $existingBook['page_count'] > 0) {
        $final_page_count = $existingBook['page_count'];
        error_log("?? Keeping existing page count: $final_page_count");
    } else if (!empty($originalPdfPath) && file_exists($originalPdfPath)) {
        // ??? ?? ??? ???? ???? ?????? ???? ????????
        $final_page_count = $calculated_page_count;
        error_log("?? Calculated page count for update: $calculated_page_count");
    }
}

// ?? FIX: ?????? ?? ?? ?????? ?????
if ($final_page_count === null || $final_page_count === '') {
    $final_page_count = 0;
} else {
    $final_page_count = intval($final_page_count);
}

error_log("?? FINAL PAGE COUNT DECISION: $final_page_count, Manual: " . ($page_count_updated ? 'YES' : 'NO'));

        error_log("?? Final Page Count: $final_page_count");

        // ?? NEW: Auto-calculate taille_before_extrait if empty
        if (empty($taille_before_extrait) && !empty($originalPdfPath) && file_exists($originalPdfPath) && $final_page_count > 0) {
            $taille_before_extrait = calculateAutoTailleBeforeExtrait($originalPdfPath, $final_page_count);
            error_log("? Auto-calculated taille_before_extrait: $taille_before_extrait");
        } else if (!empty($taille_before_extrait)) {
            error_log("?? Using manual taille_before_extrait: $taille_before_extrait");
        } else {
            error_log("?? No taille_before_extrait calculated or provided");
        }

        // Upload image (optional)
        if (!$remove_image && isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
            error_log("??? Processing NEW image upload");
            
            $uploadDir = getUploadDirectories()['images'];
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
                error_log("? Created image directory: $uploadDir");
            }

            $allowedTypes = ['image/jpeg','image/png','image/gif','image/webp'];
            if (!in_array($_FILES['image']['type'], $allowedTypes)) {
                error_log("? Invalid image type: " . $_FILES['image']['type']);
                json_error('Invalid image type');
            }

            $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
            $fileName = uniqid() . '_' . time() . '.' . $ext;
            $targetFile = getTargetPath($fileName, 'images');

            if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
                $image_url = $targetFile;
                error_log("? New image uploaded successfully: $image_url");
                
                // ??? ??? ?????? ???? ?????? ???????
                if ($id) {
                    $stmt = $pdo->prepare('SELECT image_url FROM books WHERE id = ?');
                    $stmt->execute([$id]);
                    $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($existingBook && !empty($existingBook['image_url']) && file_exists($existingBook['image_url'])) {
                        unlink($existingBook['image_url']);
                        error_log("? Deleted old book image: " . $existingBook['image_url']);
                    }
                }
            } else {
                error_log("? Failed to upload image");
                json_error('Failed to upload image');
            }
        } elseif ($id && !$remove_image && !isset($_FILES['image']) && !$use_default_image) {
            // Keep existing image if not removing and no new image uploaded and not using default image
            $stmt = $pdo->prepare('SELECT image_url FROM books WHERE id = ?');
            $stmt->execute([$id]);
            $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($existingBook) {
                $image_url = $existingBook['image_url'];
                error_log("?? Keeping existing book image: $image_url");
            }
        }

        // ?? FIX: ?????? ?? ????? ???? extracted URLs ??? ??? ?? ????? ????? ??????
        if ($id && !isset($_FILES['pdf'])) {
            error_log("?? Checking for PDF extraction updates for existing book...");
            
            // ??? ???? ???? ????? ????? ???? ?? ??? ??????? processPdfExtraction
            if ((!empty($selected_pages_toc) && empty($extracted_pdf_url)) || 
                (!empty($selected_pages_summary) && empty($extracted_summary_pdf_url))) {
                
                error_log("?? Pages specified but extraction not processed - forcing extraction...");
                
                // ??? ????? ??????
                $stmt = $pdo->prepare('SELECT original_pdf_url FROM books WHERE id = ?');
                $stmt->execute([$id]);
                $existingBook = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($existingBook && !empty($existingBook['original_pdf_url'])) {
                    $originalPdfPath = $existingBook['original_pdf_url'];
                    error_log("?? Re-extracting from original: $originalPdfPath");
                    
                    // ????? ???????
                    processPdfExtraction($originalPdfPath, basename($originalPdfPath), 
                                        $selected_pages_original, $selected_pages_toc, $selected_pages_summary, 
                                        $original_pdf_url, $pdf_url, $file_size, $extracted_pdf_url, 
                                        $extracted_file_size, $extracted_summary_pdf_url, 
                                        $extracted_summary_file_size, $id, $pdo);
                }
            }
        }

        // ?? FIX: ?????? ??????? ?? ??? extracted URLs
        error_log("?? FINAL EXTRACTED URLS CHECK:");
        error_log("   extracted_pdf_url: " . ($extracted_pdf_url ? $extracted_pdf_url : 'EMPTY'));
        error_log("   extracted_summary_pdf_url: " . ($extracted_summary_pdf_url ? $extracted_summary_pdf_url : 'EMPTY'));
        error_log("   selected_pages_toc: '$selected_pages_toc'");
        error_log("   selected_pages_summary: '$selected_pages_summary'");

        // ????? ?? ????? ????????
        if ($id) {
            error_log("?? Updating existing book ID: $id");
            
            // ???? ???? ??????? ??????????
            $updateFields = [
                'title' => $title,
                'author' => $author,
                'category_id' => $categoryId,
                'profile_category_id' => $profileCategoryId,
                'summary' => $summary,
                'table_of_contents' => $table_of_contents,
                'selected_pages_summary' => $selected_pages_summary,
                'selected_pages_toc' => $selected_pages_toc,
                'selected_pages_original' => $selected_pages_original,
                'institution' => $institution,
                'publication_year' => $publication_year,
                'keywords' => $keywords,
                'language' => $language,
                'format' => $format,
                'is_external' => $is_external,
                'is_external_plus' => $is_external_plus,
                'is_protected' => $is_protected,
                'protection_key' => $protection_key,
                'show_read_button' => $show_read_button,
                'read_button_text' => $read_button_text,
                'custom_file_size' => $custom_file_size,
                'show_file_size' => $show_file_size,
                'taille_before_extrait' => $taille_before_extrait,
                'show_taille_before_extrait' => $show_taille_before_extrait,
                'show_download_extrait_button' => $show_download_extrait_button,
                'download_extrait_button_text' => $download_extrait_button_text,
                'stockage' => $stockage,
                'original_pdf_url' => $original_pdf_url,
                'page_count' => $final_page_count,
                'external_download_link' => $external_download_link,
                'show_external_download_button' => $show_external_download_button,
                'external_download_protection_key' => $external_download_protection_key,
                'is_external_download_protected' => $is_external_download_protected,
                
                // ?? FIX CRITICAL: ??????? ????? ???????
                'is_active' => $isActive,
                'require_login_for_download' => $require_login_for_download,
                
                // ?? CRITICAL: Always update these PDF fields
                'pdf_url' => $pdf_url,
                'file_size' => $file_size,
                'extracted_pdf_url' => $extracted_pdf_url,
                'extracted_file_size' => $extracted_file_size,
                'extracted_summary_pdf_url' => $extracted_summary_pdf_url,
                'extracted_summary_file_size' => $extracted_summary_file_size,
            ];
            
            // Handle image updates
            if (!empty($image_url)) {
                $updateFields['image_url'] = $image_url;
                error_log("? Setting new image URL: $image_url");
            } elseif ($remove_image) {
                $updateFields['image_url'] = '';
                error_log("? Setting empty image URL due to removal");
            } elseif ($use_default_image) {
                $updateFields['image_url'] = $image_url;
                error_log("? Setting default image URL: $image_url");
            }
            
            // ???? ??????? SQL
            $setParts = [];
            $params = [];
            foreach ($updateFields as $field => $value) {
                $setParts[] = "$field = ?";
                $params[] = $value;
            }
            $params[] = $id;

            $sql = "UPDATE books SET " . implode(', ', $setParts) . " WHERE id = ?";
            error_log("?? SQL: $sql");
            
            // ????? ???????
            $stmt = $pdo->prepare($sql);
            $result = $stmt->execute($params);
            
            if ($result) {
                $affectedRows = $stmt->rowCount();
                error_log("? UPDATE SUCCESS - Affected rows: $affectedRows");
                error_log("? CRITICAL FIELDS UPDATED - is_active: $isActive, require_login_for_download: $require_login_for_download");
            } else {
                error_log("? UPDATE FAILED - No rows affected");
            }
            
            $message = 'Book updated successfully';
            $bookId = $id;
        } else {
            error_log("?? Inserting new book");
            
            $sql = 'INSERT INTO books (title, author, category_id, profile_category_id, image_url, summary, pdf_url, original_pdf_url, extracted_pdf_url, extracted_summary_pdf_url, file_size, extracted_file_size, extracted_summary_file_size, is_external, is_external_plus, institution, publication_year, keywords, language, format, table_of_contents, selected_pages_summary, selected_pages_toc, selected_pages_original, is_protected, protection_key, show_read_button, read_button_text, custom_file_size, show_file_size, taille_before_extrait, show_taille_before_extrait, show_download_extrait_button, download_extrait_button_text, stockage, page_count, external_download_link, show_external_download_button, external_download_protection_key, is_external_download_protected, is_active, require_login_for_download) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
            $params = [
                $title, $author, $categoryId, $profileCategoryId, $image_url, $summary, 
                $pdf_url, $original_pdf_url, $extracted_pdf_url, $extracted_summary_pdf_url, $file_size, 
                $extracted_file_size, $extracted_summary_file_size, $is_external,
                $is_external_plus,
                $institution, $publication_year, $keywords, $language, $format, $table_of_contents, 
                $selected_pages_summary, $selected_pages_toc, $selected_pages_original,
                $is_protected, $protection_key,
                $show_read_button, $read_button_text, $custom_file_size, $show_file_size,
                $taille_before_extrait, $show_taille_before_extrait,
                $show_download_extrait_button, $download_extrait_button_text,
                $stockage,
                $final_page_count,
                $external_download_link, $show_external_download_button,
                $external_download_protection_key, $is_external_download_protected,
                
                // ?? FIX CRITICAL: ??????? ????? ???????
                $isActive,
                $require_login_for_download
            ];
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            
            $bookId = $pdo->lastInsertId();
            $message = 'Book added successfully';
            
            error_log("? INSERT SUCCESS - CRITICAL FIELDS INSERTED - is_active: $isActive, require_login_for_download: $require_login_for_download");
        }

        $response = [
            'success' => true,
            'message' => $message,
            'id' => $bookId,
            'used_default_image' => $use_default_image,
            'is_protected' => $is_protected,
            'show_read_button' => $show_read_button,
            'read_button_text' => $read_button_text,
            'custom_file_size' => $custom_file_size,
            'show_file_size' => $show_file_size,
            'taille_before_extrait' => $taille_before_extrait,
            'show_taille_before_extrait' => $show_taille_before_extrait,
            'show_download_extrait_button' => $show_download_extrait_button,
            'download_extrait_button_text' => $download_extrait_button_text,
            'stockage' => $stockage,
            'page_count' => $final_page_count,
            'language' => $language,
            'format' => $format,
            'used_default_image' => $use_default_image,
            'page_count_updated' => $page_count_updated,
            'page_count_calculated' => !$page_count_updated && $final_page_count > 0,
            'image_updated' => isset($_FILES['image']) || $remove_image,
            'image_removed' => $remove_image,
            'original_pdf_preserved' => !empty($original_pdf_url),
            'original_pdf_url' => $original_pdf_url,
            'is_external' => $is_external,
            'is_external_plus' => $is_external_plus,
            'external_download_link' => $external_download_link,
            'show_external_download_button' => $show_external_download_button,
            'external_download_protection_key' => $external_download_protection_key,
            'is_external_download_protected' => $is_external_download_protected,
            
            // ?? FIX CRITICAL: ????? ????? ???????
            'is_active' => $isActive,
            'require_login_for_download' => $require_login_for_download,
        ];

        error_log("? Book saved successfully - Language: $language, Format: $format, Page Count: $final_page_count, Active: $isActive, Require Login: $require_login_for_download, Updated: " . ($page_count_updated ? 'YES' : 'NO'));
        
        json_success($message, $response);

    } catch (PDOException $e) {
        error_log("? Database error: " . $e->getMessage());
        json_error('Database error: ' . $e->getMessage());
    } catch (Exception $e) {
        error_log("? General error: " . $e->getMessage());
        json_error('Server error: ' . $e->getMessage());
    }
}

// DELETE: Remove book
if ($_SERVER['REQUEST_METHOD'] == 'DELETE') {
    $input = file_get_contents('php://input');
    parse_str($input, $deleteVars);
    $id = $deleteVars['id'] ?? null;

    if (!$id) {
        json_error('Book ID required');
    }

    try {
        // ?????? ??? ??????? ?????? ???? ???????
        $stmt = $pdo->prepare('SELECT pdf_url, original_pdf_url, extracted_pdf_url, extracted_summary_pdf_url, image_url FROM books WHERE id = ?');
        $stmt->execute([$id]);
        $book = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($book) {
            $filesToDelete = [];
            
            // ????? ????? PDF ?????
            $pdfFiles = [
                $book['pdf_url'],
                $book['original_pdf_url'],
                $book['extracted_pdf_url'],
                $book['extracted_summary_pdf_url']
            ];
            
            foreach ($pdfFiles as $pdfFile) {
                if (!empty($pdfFile) && $pdfFile !== 'uploads/no-image.jpg') {
                    $filesToDelete[] = $pdfFile;
                }
            }
            
            // ??? ??? ?????? ??????????
            if (!empty($book['image_url']) && 
                $book['image_url'] !== 'uploads/no-image.jpg' &&
                $book['image_url'] !== '../api/uploads/no-image.jpg') {
                $filesToDelete[] = $book['image_url'];
                error_log("? Added custom image to delete list: " . $book['image_url']);
            } else {
                error_log("?? Skipping default image deletion: " . ($book['image_url'] ?? 'empty'));
            }
            
            // ????? ?????
            $deletedCount = 0;
            foreach ($filesToDelete as $filePath) {
                if (!empty($filePath) && 
                    file_exists($filePath) && 
                    !filter_var($filePath, FILTER_VALIDATE_URL) &&
                    $filePath !== 'uploads/no-image.jpg' &&
                    $filePath !== '../api/uploads/no-image.jpg') {
                    
                    if (unlink($filePath)) {
                        error_log("? Deleted file: " . basename($filePath));
                        $deletedCount++;
                    } else {
                        error_log("? Failed to delete file: $filePath");
                    }
                }
            }
            
            error_log("?? Deleted $deletedCount files for book ID: $id");
        }
        
        // ??? ?????? ?? ????? ????????
        $stmt = $pdo->prepare('DELETE FROM books WHERE id = ?');
        $stmt->execute([$id]);
        
        json_success('Book deleted successfully');
    } catch(PDOException $e) {
        json_error('Database error: ' . $e->getMessage());
    }
}

// Helper functions
function getRemoteFileSize($url) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
    curl_setopt($ch, CURLOPT_HEADER, TRUE);
    curl_setopt($ch, CURLOPT_NOBODY, TRUE);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    $data = curl_exec($ch);
    $size = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return ($httpCode == 200 && $size) ? intval($size) : 0;
}

function columnExists($pdo, $table, $column) {
    try {
        $stmt = $pdo->prepare("SHOW COLUMNS FROM $table LIKE ?");
        $stmt->execute([$column]);
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

// ?? FIX: ??? ????? ??? ???? ??????? ???? ???
json_error('Method not allowed');