<?php
include 'config.php';

// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// GET: Get categories - UPDATED to support active filtering
if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    try {
        $activeOnly = isset($_GET['active_only']) && $_GET['active_only'] === 'true';
        
        if ($activeOnly) {
            error_log("📁 جلب التصنيفات النشطة فقط");
            $stmt = $pdo->query('SELECT * FROM categories WHERE is_active = TRUE ORDER BY display_order ASC, created_at DESC');
        } else {
            error_log("📁 جلب جميع التصنيفات (بما فيها غير النشطة)");
            $stmt = $pdo->query('SELECT * FROM categories ORDER BY display_order ASC, created_at DESC');
        }
        
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (!$categories) {
            $categories = [];
        }
        
        error_log("✅ إرجاع " . count($categories) . " تصنيف" . ($activeOnly ? " نشط" : ""));
        echo json_encode($categories, JSON_UNESCAPED_UNICODE);
        
    } catch(PDOException $e) {
        error_log("❌ خطأ في قاعدة البيانات: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'خطأ في قاعدة البيانات: ' . $e->getMessage()]);
    }
    exit();
}

// إضافة أو تحديث تصنيف
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        $id = $_POST['id'] ?? '';
        $name = $_POST['name'] ?? '';
        $description = $_POST['description'] ?? '';
        $displayOrder = $_POST['display_order'] ?? 0;
        
        // 🔥🔥🔥 التصحيح النهائي: معالجة حقل is_active بشكل صحيح
        $isActive = isset($_POST['is_active']) ? ($_POST['is_active'] === 'true' ? 1 : 0) : 0;
        
        $image_url = '';
        $remove_image = isset($_POST['remove_image']) && $_POST['remove_image'] === 'true';

        error_log("🔄 Processing category - ID: $id, Name: $name, Active: $isActive, Remove Image: " . ($remove_image ? 'YES' : 'NO'));

        if (empty($name)) {
            http_response_code(400);
            echo json_encode(['error' => 'اسم التصنيف مطلوب']);
            exit();
        }
        
        // 🔥 **التحقق الجديد: منع تكرار اسم التصنيف**
        error_log("🔍 Checking for duplicate category name: '$name'");
        if ($id) {
            // حالة التحديث: التحقق من التكرار مع استثناء التصنيف الحالي
            $checkStmt = $pdo->prepare('SELECT id, name FROM categories WHERE name = ? AND id != ?');
            $checkStmt->execute([$name, $id]);
        } else {
            // حالة الإضافة: التحقق من التكرار بدون استثناء
            $checkStmt = $pdo->prepare('SELECT id, name FROM categories WHERE name = ?');
            $checkStmt->execute([$name]);
        }

        $existingCategory = $checkStmt->fetch(PDO::FETCH_ASSOC);

        if ($existingCategory) {
            $errorMessage = "A category with the same name ('{$existingCategory['name']}') already exists. Please use a different name.";
            error_log("❌ Validation failed: Category with same name already exists - ID: {$existingCategory['id']}, Name: {$existingCategory['name']}");
            http_response_code(400);
            echo json_encode(['error' => $errorMessage]);
            exit();
        }
        error_log("✅ No duplicate category name found");
        
        // Handle image removal
        if ($id && $remove_image) {
            error_log("🗑️ Removing existing category image...");
            $stmt = $pdo->prepare('SELECT image_url FROM categories WHERE id = ?');
            $stmt->execute([$id]);
            $existingCategory = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingCategory && !empty($existingCategory['image_url']) && file_exists($existingCategory['image_url'])) {
                // 🔥 حماية الصورة الافتراضية من الحذف
                if ($existingCategory['image_url'] !== 'uploads/no-imagee.jpg' && 
                    $existingCategory['image_url'] !== '../api/uploads/no-imagee.jpg') {
                    if (unlink($existingCategory['image_url'])) {
                        error_log("✅ Deleted existing category image: " . $existingCategory['image_url']);
                    }
                } else {
                    error_log("ℹ️ Skipping default image deletion: " . $existingCategory['image_url']);
                }
            }
            $image_url = ''; // Set empty image URL
        }
        
        // 🔥 NEW: Auto-set default image for new categories without cover
        if (empty($id) && !isset($_FILES['image']) && !$remove_image) {
            $defaultImagePath = 'uploads/no-imagee.jpg';
            
            // التحقق من وجود الصورة الافتراضية
            if (file_exists($defaultImagePath)) {
                $image_url = $defaultImagePath;
                error_log("✅ Using default category image: $defaultImagePath");
            } else {
                error_log("⚠️ Default category image not found: $defaultImagePath");
                $image_url = '';
            }
        }
        
        // معالجة رفع الصورة - محسنة
        if (!$remove_image && isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
            $uploadDir = 'uploads/images/';
            
            // إنشاء المجلد إذا لم يكن موجوداً
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            // التحقق من نوع الملف
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            $fileType = $_FILES['image']['type'];
            
            if (!in_array($fileType, $allowedTypes)) {
                http_response_code(400);
                echo json_encode(['error' => 'نوع الملف غير مسموح به. المسموح: JPG, PNG, GIF, WEBP']);
                exit();
            }
            
            // إنشاء اسم فريد للملف
            $fileExtension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
            $fileName = uniqid() . '_' . time() . '.' . $fileExtension;
            $targetFile = $uploadDir . $fileName;
            
            if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
                $image_url = $targetFile;
                error_log("✅ New image uploaded: $image_url");
                
                // إذا كان هناك تحديث، احذف الصورة القديمة (مع حماية الصورة الافتراضية)
                if ($id) {
                    $stmt = $pdo->prepare('SELECT image_url FROM categories WHERE id = ?');
                    $stmt->execute([$id]);
                    $existingCategory = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($existingCategory && !empty($existingCategory['image_url']) && 
                        file_exists($existingCategory['image_url']) &&
                        $existingCategory['image_url'] !== 'uploads/no-imagee.jpg' &&
                        $existingCategory['image_url'] !== '../api/uploads/no-imagee.jpg') {
                        
                        unlink($existingCategory['image_url']);
                        error_log("✅ Deleted old category image: " . $existingCategory['image_url']);
                    }
                }
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'فشل في رفع الصورة']);
                exit();
            }
        } elseif ($id && !$remove_image && !isset($_FILES['image'])) {
            // Keep existing image if not removing and no new image uploaded
            $stmt = $pdo->prepare('SELECT image_url FROM categories WHERE id = ?');
            $stmt->execute([$id]);
            $existingCategory = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($existingCategory) {
                $image_url = $existingCategory['image_url'];
                error_log("📁 Keeping existing category image: $image_url");
            }
        }
        
        if (!empty($id)) {
            // تحديث التصنيف الموجود
            error_log("🔄 Updating existing category ID: $id");
            
            if (!empty($image_url)) {
                $stmt = $pdo->prepare('UPDATE categories SET name = ?, image_url = ?, description = ?, display_order = ?, is_active = ? WHERE id = ?');
                $stmt->execute([$name, $image_url, $description, $displayOrder, $isActive, $id]);
                error_log("✅ Category updated with new image and active status: $isActive");
            } else {
                $stmt = $pdo->prepare('UPDATE categories SET name = ?, description = ?, display_order = ?, is_active = ? WHERE id = ?');
                $stmt->execute([$name, $description, $displayOrder, $isActive, $id]);
                error_log("✅ Category updated without image change, active status: $isActive");
            }
            
            $lastId = $id;
            $message = 'تم تحديث التصنيف بنجاح';
        } else {
            // إضافة تصنيف جديد
            error_log("🆕 Adding new category with active status: $isActive");
            $stmt = $pdo->prepare('INSERT INTO categories (name, image_url, description, display_order, is_active) VALUES (?, ?, ?, ?, ?)');
            $stmt->execute([$name, $image_url, $description, $displayOrder, $isActive]);
            
            $lastId = $pdo->lastInsertId();
            $message = 'تم إضافة التصنيف بنجاح';
            error_log("✅ New category added with ID: $lastId and active status: $isActive");
        }
        
        echo json_encode([
            'success' => true,
            'message' => $message, 
            'id' => $lastId,
            'name' => $name,
            'description' => $description,
            'display_order' => $displayOrder,
            'is_active' => $isActive,
            'image_url' => $image_url,
            'image_removed' => $remove_image,
            'used_default_image' => ($image_url === 'uploads/no-imagee.jpg') // 🔥 NEW: للإشارة لاستخدام الصورة الافتراضية
        ], JSON_UNESCAPED_UNICODE);
        
    } catch(PDOException $e) {
        error_log("❌ Database error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'خطأ في قاعدة البيانات: ' . $e->getMessage()]);
    }
    exit();
}

// حذف تصنيف
if ($_SERVER['REQUEST_METHOD'] == 'DELETE') {
    $input = file_get_contents('php://input');
    parse_str($input, $deleteVars);
    $id = $deleteVars['id'] ?? null;
    
    if (!$id) {
        http_response_code(400);
        echo json_encode(['error' => 'معرف التصنيف مطلوب']);
        exit();
    }
    
    try {
        // الحصول على معلومات التصنيف لحذف الصورة
        $stmt = $pdo->prepare('SELECT image_url FROM categories WHERE id = ?');
        $stmt->execute([$id]);
        $category = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // 🔥 حذف الصورة من السيرفر مع حماية الصورة الافتراضية
        if ($category && !empty($category['image_url']) && file_exists($category['image_url'])) {
            // 🔥 حماية الصورة الافتراضية من الحذف
            if ($category['image_url'] !== 'uploads/no-imagee.jpg' && 
                $category['image_url'] !== '../api/uploads/no-imagee.jpg') {
                
                if (unlink($category['image_url'])) {
                    error_log("✅ Deleted custom category image: " . $category['image_url']);
                } else {
                    error_log("❌ Failed to delete category image: " . $category['image_url']);
                }
            } else {
                error_log("ℹ️ Skipping default image deletion: " . $category['image_url']);
            }
        }
        
        $stmt = $pdo->prepare('DELETE FROM categories WHERE id = ?');
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'تم حذف التصنيف بنجاح']);
        
    } catch(PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'خطأ في قاعدة البيانات: ' . $e->getMessage()]);
    }
    exit();
}

http_response_code(405);
echo json_encode(['error' => 'طريقة الطلب غير مسموحة']);
?>