<?php
include 'config.php';

header('Content-Type: application/json; charset=utf-8');

// تفعيل التصحيح
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/debug.log');

error_log("=== DEBUG: contact_settings.php accessed ===");

// GET: جلب إعدادات الاتصال
if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    try {
        error_log("📞 GET Request for contact settings");
        
        $stmt = $pdo->query('SELECT * FROM contact_settings ORDER BY id DESC LIMIT 1');
        $settings = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$settings) {
            error_log("⚠️ No contact settings found, returning defaults");
            // إرجاع إعدادات افتراضية إذا لم توجد في قاعدة البيانات
            $settings = [
                'phone_number' => '+212612345678',
                'whatsapp_number' => '+212612345678',
                'email' => 'admin@watchevents.shop',
                'company_name' => 'WatchEvents Administration',
                'company_address' => 'Casablanca, Morocco', // 🔥 NEW: Default address
                'business_hours' => "Monday - Friday: 09:00 - 18:00\nSaturday: 10:00 - 16:00\nSunday: Closed"
            ];
        }
        
        error_log("✅ Contact settings retrieved successfully");
        echo json_encode($settings, JSON_UNESCAPED_UNICODE);
        
    } catch (PDOException $e) {
        error_log("❌ Database error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
    exit();
}

// POST: حفظ أو تحديث إعدادات الاتصال
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        error_log("=== CONTACT SETTINGS POST REQUEST ===");
        
        // قراءة البيانات من POST
        $phone_number = $_POST['phone_number'] ?? '';
        $whatsapp_number = $_POST['whatsapp_number'] ?? '';
        $email = $_POST['email'] ?? '';
        $company_name = $_POST['company_name'] ?? '';
        $company_address = $_POST['company_address'] ?? ''; // 🔥 NEW: Address field
        $business_hours = $_POST['business_hours'] ?? '';
        
        error_log("📝 Processing contact settings:");
        error_log("   Phone: $phone_number");
        error_log("   WhatsApp: $whatsapp_number");
        error_log("   Email: $email");
        error_log("   Company: $company_name");
        error_log("   Address: $company_address"); // 🔥 NEW
        error_log("   Business Hours: $business_hours");
        
        // التحقق من البيانات المطلوبة
        if (empty($phone_number) || empty($whatsapp_number) || empty($email) || empty($company_name)) {
            error_log("❌ Validation failed: Required fields missing");
            http_response_code(400);
            echo json_encode(['error' => 'All fields are required']);
            exit();
        }
        
        // التحقق من صحة البريد الإلكتروني
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            error_log("❌ Invalid email format: $email");
            http_response_code(400);
            echo json_encode(['error' => 'The email format is invalid.']);
            exit();
        }
        
        // التحقق من وجود إعدادات سابقة
        $checkStmt = $pdo->query('SELECT COUNT(*) FROM contact_settings');
        $settingsExist = $checkStmt->fetchColumn() > 0;
        
        if ($settingsExist) {
            // تحديث الإعدادات الموجودة
            error_log("🔄 Updating existing contact settings");
            $stmt = $pdo->prepare('UPDATE contact_settings SET 
                phone_number = ?, 
                whatsapp_number = ?, 
                email = ?, 
                company_name = ?, 
                company_address = ?,  -- 🔥 NEW: Address field
                business_hours = ?,
                updated_at = CURRENT_TIMESTAMP 
                ORDER BY id DESC LIMIT 1');
        } else {
            // إضافة إعدادات جديدة
            error_log("🆕 Inserting new contact settings");
            $stmt = $pdo->prepare('INSERT INTO contact_settings 
                (phone_number, whatsapp_number, email, company_name, company_address, business_hours) 
                VALUES (?, ?, ?, ?, ?, ?)');
        }
        
        $stmt->execute([$phone_number, $whatsapp_number, $email, $company_name, $company_address, $business_hours]);
        
        $response = [
            'success' => true,
            'message' => 'Contact settings saved successfully',
            'data' => [
                'phone_number' => $phone_number,
                'whatsapp_number' => $whatsapp_number,
                'email' => $email,
                'company_name' => $company_name,
                'company_address' => $company_address, // 🔥 NEW
                'business_hours' => $business_hours
            ]
        ];
        
        error_log("✅ Contact settings saved successfully with address");
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        
    } catch (PDOException $e) {
        error_log("❌ Database error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
    exit();
}

// إذا وصلنا هنا، الطريقة غير مسموحة
http_response_code(405);
echo json_encode(['error' => 'Method not allowed']);
?>