<?php
include 'config.php';

// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// GET: Get profile categories - UPDATED to support active filtering
if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    try {
        error_log("📖 طلب الحصول على تصنيفات البروفايل");
        
        $activeOnly = isset($_GET['active_only']) && $_GET['active_only'] === 'true';
        
        if ($activeOnly) {
            error_log("📁 جلب تصنيفات البروفايل النشطة فقط");
            $stmt = $pdo->query('SELECT * FROM profile_categories WHERE is_active = TRUE ORDER BY display_order ASC, created_at DESC');
        } else {
            error_log("📁 جلب جميع تصنيفات البروفايل (بما فيها غير النشطة)");
            $stmt = $pdo->query('SELECT * FROM profile_categories ORDER BY display_order ASC, created_at DESC');
        }
        
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        error_log("✅ إرجاع " . count($categories) . " تصنيف بروفايل" . ($activeOnly ? " نشط" : ""));
        echo json_encode($categories, JSON_UNESCAPED_UNICODE);
        
    } catch(PDOException $e) {
        error_log("💥 خطأ في قاعدة البيانات: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'خطأ في قاعدة البيانات: ' . $e->getMessage()]);
    }
    exit();
}

// POST: Add or Update profile category
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        $id = $_POST['id'] ?? '';
        $name = $_POST['name'] ?? '';
        $description = $_POST['description'] ?? '';
        $displayOrder = $_POST['display_order'] ?? 0;
        
        // 🔥🔥🔥 التصحيح النهائي: معالجة حقل is_active بشكل صحيح
        $isActive = isset($_POST['is_active']) ? ($_POST['is_active'] === 'true' ? 1 : 0) : 0;
        
        $image_url = '';
        $remove_image = isset($_POST['remove_image']) && $_POST['remove_image'] === 'true';

        error_log("➕ طلب معالجة تصنيف بروفايل - ID: $id, Name: $name, Active: $isActive");

        if (empty($name)) {
            http_response_code(400);
            echo json_encode(['error' => 'اسم التصنيف مطلوب']);
            exit();
        }
        
        // 🔥 **التحقق الجديد: منع تكرار اسم تصنيف البروفايل**
        error_log("🔍 Checking for duplicate profile category name: '$name'");
        if ($id) {
            // حالة التحديث: التحقق من التكرار مع استثناء التصنيف الحالي
            $checkStmt = $pdo->prepare('SELECT id, name FROM profile_categories WHERE name = ? AND id != ?');
            $checkStmt->execute([$name, $id]);
        } else {
            // حالة الإضافة: التحقق من التكرار بدون استثناء
            $checkStmt = $pdo->prepare('SELECT id, name FROM profile_categories WHERE name = ?');
            $checkStmt->execute([$name]);
        }

        $existingCategory = $checkStmt->fetch(PDO::FETCH_ASSOC);

        if ($existingCategory) {
            $errorMessage = "A profile category with the same name ('{$existingCategory['name']}') already exists. Please use a different name.";
            error_log("❌ Validation failed: Profile category with same name already exists - ID: {$existingCategory['id']}, Name: {$existingCategory['name']}");
            http_response_code(400);
            echo json_encode(['error' => $errorMessage]);
            exit();
        }
        error_log("✅ No duplicate profile category name found");
        
        // Handle image removal
        if ($id && $remove_image) {
            error_log("🗑️ Removing existing profile category image...");
            $stmt = $pdo->prepare('SELECT image_url FROM profile_categories WHERE id = ?');
            $stmt->execute([$id]);
            $existingCategory = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingCategory && !empty($existingCategory['image_url']) && file_exists($existingCategory['image_url'])) {
                // 🔥 حماية الصورة الافتراضية من الحذف
                if ($existingCategory['image_url'] !== 'uploads/no-imagee.jpg' && 
                    $existingCategory['image_url'] !== '../api/uploads/no-imagee.jpg') {
                    if (unlink($existingCategory['image_url'])) {
                        error_log("✅ Deleted existing profile category image: " . $existingCategory['image_url']);
                    }
                } else {
                    error_log("ℹ️ Skipping default image deletion: " . $existingCategory['image_url']);
                }
            }
            $image_url = ''; // Set empty image URL
        }
        
        // 🔥 NEW: Auto-set default image for new profile categories without cover
        if (empty($id) && !isset($_FILES['image']) && !$remove_image) {
            $defaultImagePath = 'uploads/no-imagee.jpg';
            
            // التحقق من وجود الصورة الافتراضية
            if (file_exists($defaultImagePath)) {
                $image_url = $defaultImagePath;
                error_log("✅ Using default profile category image: $defaultImagePath");
            } else {
                error_log("⚠️ Default profile category image not found: $defaultImagePath");
                $image_url = '';
            }
        }
        
        // Handle image upload
        if (!$remove_image && isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
            $uploadDir = 'uploads/profile_categories/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            if (!in_array($_FILES['image']['type'], $allowedTypes)) {
                http_response_code(400);
                echo json_encode(['error' => 'نوع ملف الصورة غير مسموح به']);
                exit();
            }
            
            $fileExtension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
            $fileName = uniqid() . '_' . time() . '.' . $fileExtension;
            $targetFile = $uploadDir . $fileName;
            
            if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
                $image_url = $targetFile;
                error_log("✅ New profile category image uploaded: $image_url");
                
                // إذا كان هناك تحديث، احذف الصورة القديمة (مع حماية الصورة الافتراضية)
                if ($id) {
                    $stmt = $pdo->prepare('SELECT image_url FROM profile_categories WHERE id = ?');
                    $stmt->execute([$id]);
                    $existingCategory = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($existingCategory && !empty($existingCategory['image_url']) && 
                        file_exists($existingCategory['image_url']) &&
                        $existingCategory['image_url'] !== 'uploads/no-imagee.jpg' &&
                        $existingCategory['image_url'] !== '../api/uploads/no-imagee.jpg') {
                        
                        unlink($existingCategory['image_url']);
                        error_log("✅ Deleted old profile category image: " . $existingCategory['image_url']);
                    }
                }
            }
        } elseif ($id && !$remove_image && !isset($_FILES['image'])) {
            // Keep existing image if not removing and no new image uploaded
            $stmt = $pdo->prepare('SELECT image_url FROM profile_categories WHERE id = ?');
            $stmt->execute([$id]);
            $existingCategory = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($existingCategory) {
                $image_url = $existingCategory['image_url'];
                error_log("📁 Keeping existing profile category image: $image_url");
            }
        }
        
        if (!empty($id)) {
            // تحديث التصنيف الموجود
            error_log("🔄 Updating existing profile category ID: $id with active status: $isActive");
            
            if (!empty($image_url)) {
                $stmt = $pdo->prepare('UPDATE profile_categories SET name = ?, image_url = ?, description = ?, display_order = ?, is_active = ? WHERE id = ?');
                $stmt->execute([$name, $image_url, $description, $displayOrder, $isActive, $id]);
            } else {
                $stmt = $pdo->prepare('UPDATE profile_categories SET name = ?, description = ?, display_order = ?, is_active = ? WHERE id = ?');
                $stmt->execute([$name, $description, $displayOrder, $isActive, $id]);
            }
            
            $lastId = $id;
            $message = 'تم تحديث التصنيف بنجاح';
        } else {
            // إضافة تصنيف جديد
            error_log("🆕 Adding new profile category with active status: $isActive");
            $stmt = $pdo->prepare('INSERT INTO profile_categories (name, image_url, description, display_order, is_active) VALUES (?, ?, ?, ?, ?)');
            $stmt->execute([$name, $image_url, $description, $displayOrder, $isActive]);
            
            $lastId = $pdo->lastInsertId();
            $message = 'تم إضافة التصنيف بنجاح';
        }
        
        error_log("✅ Profile category saved successfully: $name (ID: $lastId, Active: $isActive)");
        
        echo json_encode([
            'success' => true,
            'message' => $message,
            'id' => $lastId,
            'is_active' => $isActive,
            'image_url' => $image_url,
            'image_removed' => $remove_image,
            'used_default_image' => ($image_url === 'uploads/no-imagee.jpg') // 🔥 NEW: للإشارة لاستخدام الصورة الافتراضية
        ], JSON_UNESCAPED_UNICODE);
        
    } catch(PDOException $e) {
        error_log("💥 خطأ في قاعدة البيانات: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'خطأ في قاعدة البيانات: ' . $e->getMessage()]);
    }
    exit();
}

// DELETE: Remove profile category
if ($_SERVER['REQUEST_METHOD'] == 'DELETE') {
    $input = file_get_contents('php://input');
    parse_str($input, $deleteVars);
    $id = $deleteVars['id'] ?? null;
    
    if (!$id) {
        http_response_code(400);
        echo json_encode(['error' => 'معرف التصنيف مطلوب']);
        exit();
    }
    
    try {
        // الحصول على معلومات التصنيف لحذف الصورة
        $stmt = $pdo->prepare('SELECT image_url FROM profile_categories WHERE id = ?');
        $stmt->execute([$id]);
        $category = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // 🔥 حذف الصورة من السيرفر مع حماية الصورة الافتراضية
        if ($category && !empty($category['image_url']) && file_exists($category['image_url'])) {
            // 🔥 حماية الصورة الافتراضية من الحذف
            if ($category['image_url'] !== 'uploads/no-imagee.jpg' && 
                $category['image_url'] !== '../api/uploads/no-imagee.jpg') {
                
                if (unlink($category['image_url'])) {
                    error_log("✅ Deleted custom profile category image: " . $category['image_url']);
                } else {
                    error_log("❌ Failed to delete profile category image: " . $category['image_url']);
                }
            } else {
                error_log("ℹ️ Skipping default image deletion: " . $category['image_url']);
            }
        }
        
        $stmt = $pdo->prepare('DELETE FROM profile_categories WHERE id = ?');
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'تم حذف التصنيف بنجاح']);
        
    } catch(PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'خطأ في قاعدة البيانات: ' . $e->getMessage()]);
    }
    exit();
}

http_response_code(405);
echo json_encode(['error' => 'طريقة الطلب غير مسموحة']);
?>